#ifndef VECTOR_MAP_H
#define VECTOR_MAP_H

#include <vector>
#include <iostream>

namespace hugo {

  /** The VectorMapFactory template class is a factory class
   *  to create maps for the edge and nodes. This map factory
   *  use the std::vector to implement the container function.
   *
   *  The template parameter is the MapRegistry that the maps
   *  will belong to.
   */
	
  template <typename MapRegistry>
  class VectorMapFactory {
  public:
		
    /// The graph type of the maps. 
    typedef typename MapRegistry::Graph Graph;
    /// The key type of the maps.
    typedef typename MapRegistry::Key Key;
    /// The iterator to iterate on the keys.
    typedef typename MapRegistry::KeyIt KeyIt;

    /// The MapBase of the Map which imlements the core regisitry function.
    typedef typename MapRegistry::MapBase MapBase;

		
    /** The template Map type.
     */
    template <typename V> 
    class Map : public MapBase {
    public:

      /// The value type of the map.
      typedef V Value;

      typedef std::vector<Value> Container;	

      /** Default constructor for the map.
       */
      Map() {}
		
      /** Graph and Registry initialized map constructor.
       */
      Map(const Graph& g, MapRegistry& r) : MapBase(g, r) {
	init();
      }

      /** Constructor to use default value to initialize the map. 
       */
      Map(const Graph& g, MapRegistry& r, const Value& v) : MapBase(g, r) {
	init();
	for (KeyIt it(*getGraph()); getGraph()->valid(it); getGraph()->next(it)) {
          set(it, v);
        }
      }

      /** Constructor to copy a map of an other map type.
       */
      template <typename CMap> Map(const CMap& copy) : MapBase(copy) {
	if (getGraph()) {
	  init();
	  for (KeyIt it(*getGraph()); getGraph()->valid(it); getGraph()->next(it)) {
	    set(it, copy[it]);
	  }
	}
      }

      /** Assign operator to copy a map an other map type.
       */
      template <typename CMap> Map& operator=(const CMap& copy) {
	if (getGraph()) {
	  destroy();
	} 
	this->MapBase::operator=(copy);
	if (getGraph()) {
	  init();
	  for (KeyIt it(*getGraph()); getGraph()->valid(it); getGraph()->next(it)) {
	    set(it, copy[it]);
	  }
	}
      }

      /** The destructor of the map.
       */
      virtual ~Map() {
	destroy();
      }
		
      /**
       * The subscript operator. The map can be subscripted by the
       * actual keys of the graph. 
       */
      typename Container::reference operator[](const Key& key) {
	int id = getGraph()->id(key);
	return container[id];
      } 
		
      /**
       * The const subscript operator. The map can be subscripted by the
       * actual keys of the graph. 
       */
      typename Container::const_reference operator[](const Key& key) const {
	int id = getGraph()->id(key);
	return container[id];
      }

      /** Setter function of the map. Equivalent with map[key] = val.
       *  This is a compatibility feature with the not dereferable maps.
       */
      void set(const Key& key, const Value& val) {
	int id = getGraph()->id(key);
	container[id] = val;
      }
		
      /** Add a new key to the map. It called by the map registry.
       */
      void add(const Key& key) {
	int id = getGraph()->id(key);
	if (id >= container.size()) {
	  container.resize(id + 1);
	}
      }
		
      /** Erease a key from the map. It called by the map registry.
       */
      void erase(const Key& key) {}

      /** Compatible iterator with the stl maps' iterators.
       *  It iterates on pairs of a key and a value.
       */
      class iterator {
	friend class Map;
	friend class const_iterator;
      private:

	/** Private constructor to initalize the the iterators returned
	 *  by the begin() and end().
	 */
	iterator (Map& pmap, const KeyIt& pit) : map(&pmap), it(pit) {}

      public:

	/** Default constructor. 
	 */
	iterator() {}

	/** Dereference operator for map.
	 */	 
	std::pair<const Key, Value> operator*() {
	  return std::pair<const Key, Value>(it, (*map)[it]);
	}

	/** Arrow operator for map.
	 */	 
	std::pair<const Key, Value>* operator->() {
	  static std::pair<const Key, Value> tmp = operator*();
	  return &tmp;
	}

	/** The pre increment operator of the map.
	 */
	iterator& operator++() { 
	  map->getGraph()->next(it); 
	  return *this; 
	}

	/** The post increment operator of the map.
	 */
	iterator operator++(int) { 
	  iterator tmp(it); 
	  map.getGraph()->next(it); 
	  return tmp; 
	}

	/** The equality operator of the map.
	 */
	bool operator==(const_iterator p_it) {
	  return p_it.it == it;
	}
	
	/** The not-equality operator of the map.
	 */
	bool operator!=(const_iterator p_it) {
	  return !(*this == p_it);
	}
	
      private:
	Map* map;
	KeyIt it;
      };

      /** Returns the begin iterator of the map.
       */
      iterator begin() {
	return iterator(*this, KeyIt(*getGraph()));
      }

      /** Returns the end iterator of the map.
       */
      iterator end() {
	return iterator(*this, INVALID);
      }

      class const_iterator {
	friend class Map;
	friend class iterator;
      private:

	/** Private constructor to initalize the the iterators returned
	 *  by the begin() and end().
	 */
	const_iterator (Map& pmap, const KeyIt& pit) : map(&pmap), it(pit) {}

      public:

	/** Default constructor. 
	 */
	const_iterator() {}

	/** Constructor to convert iterator to const_iterator.
	 */
	const_iterator(iterator p_it) {
	  it = p_it.it;
	}
      
	/** Dereference operator for map.
	 */	 
	std::pair<const Key, const Value> operator*() const {
	  return std::pair<const Key, const Value>(it, (*map)[it]);
	}

	/** Arrow operator for map.
	 */	 
	std::pair<const Key, const Value>* operator->() const {
	  static std::pair<const Key, const Value> tmp = operator*();
	  return &tmp;
	}

	/** The pre increment operator of the map.
	 */
	const_iterator& operator++() { 
	  map->getGraph()->next(it); 
	  return *this; 
	}

	/** The post increment operator of the map.
	 */
	const_iterator operator++(int) { 
	  const_iterator tmp(it); 
	  map->getGraph()->next(it); 
	  return tmp; 
	}

	/** The equality operator of the map.
	 */
	bool operator==(const_iterator p_it) {
	  return p_it.it == it;
	}
	
	/** The not-equality operator of the map.
	 */
	bool operator!=(const_iterator p_it) {
	  return !(*this == p_it);
	}
	
      private:
	const Map* map;
	KeyIt it;
      };

      /** Returns the begin const_iterator of the map.
       */
      const_iterator begin() const {
	return const_iterator(*this, KeyIt(*getGraph()));
      }

      /** Returns the end const_iterator of the map.
       */
      const_iterator end() const {
	return const_iterator(*this, INVALID);
      }

      private:
		
      Container container;

    };
		
  };

}

#endif
