// -*- c++ -*-
#ifndef HUGO_BOUNDINGBOX_H
#define HUGO_BOUNDINGBOX_H


#include <xy.h>

namespace hugo {

  /** \brief
     Implementation of a bounding box of plainvectors.
     
  */
  template<typename T>
    class BoundingBox {
      xy<T> bottomleft, topright;
      bool _empty;
    public:
      
      ///Default constructor: an empty bounding box
      BoundingBox() { _empty = true; }

      ///Constructing the instance from one point
      BoundingBox(xy<T> a) { bottomleft=topright=a; _empty = false; }

      ///Is there any point added
      bool empty() const {
	return _empty;
      }

      ///Gives back the bottom left corner (if the bounding box is empty, then the return value is not defined) 
      xy<T> bottomLeft() const {
	return bottomleft;
      };

      ///Gives back the top right corner (if the bounding box is empty, then the return value is not defined) 
      xy<T> topRight() const {
	return topright;
      };

      ///Checks whether a point is inside a bounding box
      bool inside(const xy<T>& u){
	if (_empty)
	  return false;
	else{
	  return ((u.x-bottomleft.x)*(topright.x-u.x) >= 0 &&
		  (u.y-bottomleft.y)*(topright.y-u.y) >= 0 );
	}
      }
  
      ///Increments a bounding box with a point
      BoundingBox& operator +=(const xy<T>& u){
	if (_empty){
	  bottomleft=topright=u;
	  _empty = false;
	}
	else{
	  if (bottomleft.x > u.x) bottomleft.x = u.x;
	  if (bottomleft.y > u.y) bottomleft.y = u.y;
	  if (topright.x < u.x) topright.x = u.x;
	  if (topright.y < u.y) topright.y = u.y;
	}
	return *this;
      };
  
      ///Sums a bounding box and a point
      BoundingBox operator +(const xy<T>& u){
	BoundingBox b = *this;
	return b += u;
      };

      ///Increments a bounding box with an other bounding box
      BoundingBox& operator +=(const BoundingBox &u){
	if ( !u.empty() ){
	  *this += u.bottomLeft();
	  *this += u.topRight();
	}
	return *this;
      };
  
      ///Sums two bounding boxes
      BoundingBox operator +(const BoundingBox& u){
	BoundingBox b = *this;
	return b += u;
      };

    };//class BoundingBox


} //namespace hugo

#endif //HUGO_BOUNDINGBOX_H
