// -*- C++ -*- //

#ifndef MAIN_WIN_H
#define MAIN_WIN_H

#include "all_include.h"
#include "algowin.h"
#include "map_win.h"
#include "new_map_win.h"
#include "nbtab.h"
#include <libgnomecanvasmm.h>
#include <libgnomecanvasmm/polygon.h>


///This class is the main window of GUI.

///It has menus, and a notebook. Notebook has different pages,
///the so called tabs (\ref NoteBookTab). Each \ref NoteBookTab contains a canvas on which graphs can be drawn.
///To manage creation and close of tabs and tabswitching is the task of MainWin.
class MainWin : public Gtk::Window
{
  ///Container in which the menus and the notebook is.
  Gtk::VBox vbox;

  ///The notebook that has tabs (\ref NoteBookTab) with different graphs.
  Gtk::Notebook notebook;

  ///The tool selected to manipulate graph.

  ///It has to be stored, because in case of tabswitching
  ///the correct tool has to be set for the actual graph.
  int active_tool;

  ///The number of active tab in the notebook.
  int active_tab;

  ///Vector of existing tabs in the notebook.
  std::vector<NoteBookTab *> tabs;

  ///Vector of the name of tabs.

  ///All \ref NoteBookTab has a name that is stored here. The index of the name
  ///is the same as the index of the \ref NoteBookTab in \ref tabs.
  std::vector<std::string> tabnames;

  ///Counter of occurence of the same file names.

  ///If a file is opened more than once we have to score
  ///the occurences to let the titles on tabs different.
  ///If more than one occurence is present, from the second
  ///one near the filename the number of the occurence appear.
  std::map<std::string, int> strinst;

  ///Set of opened \ref AlgoWin s.

  ///More than one \refAlgoWin can be opened. We have to
  ///communicate with them in case of new \ref NoteBookTab creation,
  ///\ref NoteBookTab close, or map change. Therefore we have to score
  ///their occurences.
  std::set< AlgoWin* > aws;

public:
  
  ///Constructor of the \ref MainWin.

  ///It creates the menus, the toolbar and the notebook in which
  ///\ref NoteBookTab s take place. \ref NoteBookTab s are the
  ///holder of the canvases on which the graphs are drawn.
  MainWin();

  ///Sets title of tabs.

  ///It alse registrates it in \ref tabnames. If more than one
  ///occurence is in the notebook of the same file it has to
  ///extend tabname with the number of occurence.
  void set_tabtitle(std::string);

  ///ActionGroup for menu
  Glib::RefPtr<Gtk::ActionGroup> ag;

  ///UIManager for menu
  Glib::RefPtr<Gtk::UIManager> uim;

  ///Creates a new \ref NoteBookTab and opens the given file.

  ///It is called only with command line parameters at stratup.
  void readFile(const std::string &);

  ///Tooltips
  Gtk::Tooltips tooltips;

  //Call-backs of buttons

  ///Callback for 'FileNew' action.
  virtual void newFile();
  ///Callback for 'FileOpen' action.
  virtual void openFile();
  ///Callback for 'FileSave' action.
  virtual void saveFile();
  ///Callback for 'FileSaveAs' action.
  virtual void saveFileAs();
  ///Callback for 'Close' action.
  virtual void close();

  //Toolbar

  ///Callback for 'zoomIn' action.

  ///It calls the appropriate function in
  ///\ref GraphDisplayerCanvas
  virtual void zoomIn();
  ///Callback for 'zoomOut' action.

  ///It calls the appropriate function in
  ///\ref GraphDisplayerCanvas
  virtual void zoomOut();
  ///Callback for 'zoomFit' action.

  ///It calls the appropriate function in
  ///\ref GraphDisplayerCanvas
  virtual void zoomFit();
  ///Callback for 'zoom100' action.

  ///It calls the appropriate function in
  ///\ref GraphDisplayerCanvas
  virtual void zoom100();

  ///Callback for Show Maps menupoint.

  ///It calls the appropriate function in
  ///\ref NoteBookTab
  virtual void createMapWin();

  ///Pops up an Algorithm window.

  ///It not only creates but registrates the newly created \ref AlgoWin.
  ///It is necessary, because in case of changement between tabs or maps
  ///we have to communicate with it. Signals are also have to be connected
  ///to it, because \ref AlgoWin emits signals if it needs anything (maplist, deregistration).
  ///\param algo type of the algorithm to run.
  virtual void createAlgoWin(int algo);

  ///Deregisters AlgoWin

  ///This is the function connected to the closing signal of \ref AlgoWin.
  ///It only deletes the sender \ref AlgoWin from \ref aws. This function
  ///is called only by the closing \ref AlgoWin itself.
  ///\param aw the \ref AlgoWin to delete.
  virtual void deRegisterAlgoWin(AlgoWin * aw);

  ///Updates list of tabs in all of the \ref AlgoWin

  ///When \ref NoteBookTab inserted somewhere or closed one tablist in all \ref AlgoWin
  ///have to be updated. That is why we score all the opened \ref AlgoWin.
  ///During update \ref tabnames will be passed to each \ref AlgoWin.
  virtual void updateAlgoWinTabs();

  ///Refresh list of maps in the AlgoWin that requested it.

  ///In an \ref AlgoWin there is a ComboBoxText, in which
  ///a \ref NoteBookTab can be chosen that contains the graph and the maps,
  ///on which we would like to run algorithms. If we change the
  ///tab the available maps also have to be updated, because
  ///in the different tabs different maps are available. Therefore
  ///on tab change the \ref AlgoWin emits a signal that contains itself
  ///so that the appropriate maps can be sent to it. For the sake of simplicity
  ///the program answers this call with the mapstorage of the newly selected tab.
  ///\param aw the caller \ref AlgoWin
  ///\param tabname the newly selected tab in the \ref AlgoWin
  virtual void updateAlgoWinMaps(AlgoWin * aw, std::string tabname);

  ///Registrates the new graph-editor tool in hand.

  ///The editor-tool in hand is global, it is the same for all tab
  ///at the same time. Therefore the active tool has to be scored here (\ref active_tool).
  ///This function is the callback function of the editor-tool buttons. It sets \ref active_tool
  ///to the correct value.
  ///\param tool the newly selected graph-editor tool (See all_include.h)
  virtual void changeEditorialTool(int tool);

  ///Pops up a \ref NewMapWin dialog after requested by a \ref MapWin

  ///Each tab can pop-up a \ref MapWin. In \ref MapWin new tab can be created.
  ///In this case \ref NoteBookTab emits a signal. This function is connected to that signal.
  ///It sends the caller \ref NoteBookTab and whether an edgemap or a nodemap should be created.
  ///Caller \ref NoteBookTab is necessary for the window to be able to place the new map in its
  ///correct place.
  ///\param nbt the caller tab
  ///\param itisedge true if edgemap has to be created, false if nodemap
  virtual void createNewMapWinAfterSignal(NoteBookTab * nbt, bool itisedge);

  ///Pops up a \ref NewMapWin dialog after requested by an \ref AlgoWin

  ///\ref AlgoWin can also can request a \ref NewMapWin to pop-up. 
  ///It emits a signal in this case. This function is bound to that signal.
  ///The signal contains the name of \ref NoteBookTab in which the new map has to be
  ///placed and whether the new map is an edgemap or a nodemap.
  ///\ref tabname the tab in which the new map has to be placed
  ///\ref itisedge true if the new map will be edge map, false if it will be nodemap
  virtual void createNewMapWinTabString(std::string tabname, bool itisedge);

  ///Pops up a \ref NewMapWin dialog if button on \ref MainWin has been pressed.

  ///In this case a general \ref NewMapWin will be popped up. This means that
  ///both edge and nodemap can be created by it. The new map will be placed in
  ///\MapStorage of the actual selected \ref NoteBookTab.
  virtual void createNewMapWin();

  //Notebook handlers
  ///Callback for 'FileNewTab' action.
  virtual void newTab();

  ///Callback for 'FileCloseTab' action.

  ///It closes the actual \ref NoteBookTab and registrates this event:
  ///data is shifted to the correct places in vectors.
  virtual void closeTab();

  ///Tabswitching handler

  ///Sets the variables that have to store the actual state, and it
  ///updates the title of window to the actually selected \ref NoteBookTab.
  virtual void onChangeTab(GtkNotebookPage*, guint);
};

#endif //MAIN_WIN_H
