// -*- c++ -*-
#ifndef HUGO_MINCOSTFLOW_H
#define HUGO_MINCOSTFLOW_H

///\ingroup galgs
///\file
///\brief An algorithm for finding a flow of value \c k (for small values of \c k) having minimal total cost 


#include <hugo/dijkstra.h>
#include <hugo/graph_wrapper.h>
#include <hugo/maps.h>
#include <vector>
#include <for_each_macros.h>

namespace hugo {

/// \addtogroup galgs
/// @{

  ///\brief Implementation of an algorithm for finding a flow of value \c k 
  ///(for small values of \c k) having minimal total cost between 2 nodes 
  /// 
  ///
  /// The class \ref hugo::MinCostFlow "MinCostFlow" implements
  /// an algorithm for solving the following general minimum cost flow problem>
  /// 
  ///
  ///
  /// \warning It is assumed here that the problem has a feasible solution
  ///
  /// The range of the length (weight) function is nonnegative reals but 
  /// the range of capacity function is the set of nonnegative integers. 
  /// It is not a polinomial time algorithm for counting the minimum cost
  /// maximal flow, since it counts the minimum cost flow for every value 0..M
  /// where \c M is the value of the maximal flow.
  ///
  ///\author Attila Bernath
  template <typename Graph, typename LengthMap, typename SupplyMap>
  class MinCostFlow {

    typedef typename LengthMap::ValueType Length;


    typedef typename SupplyMap::ValueType Supply;
    
    typedef typename Graph::Node Node;
    typedef typename Graph::NodeIt NodeIt;
    typedef typename Graph::Edge Edge;
    typedef typename Graph::OutEdgeIt OutEdgeIt;
    typedef typename Graph::template EdgeMap<int> EdgeIntMap;

    //    typedef ConstMap<Edge,int> ConstMap;

    typedef ResGraphWrapper<const Graph,int,CapacityMap,EdgeIntMap> ResGraphType;
    typedef typename ResGraphType::Edge ResGraphEdge;

    class ModLengthMap {   
      //typedef typename ResGraphType::template NodeMap<Length> NodeMap;
      typedef typename Graph::template NodeMap<Length> NodeMap;
      const ResGraphType& G;
      //      const EdgeIntMap& rev;
      const LengthMap &ol;
      const NodeMap &pot;
    public :
      typedef typename LengthMap::KeyType KeyType;
      typedef typename LengthMap::ValueType ValueType;
	
      ValueType operator[](typename ResGraphType::Edge e) const {     
	if (G.forward(e))
	  return  ol[e]-(pot[G.head(e)]-pot[G.tail(e)]);   
	else
	  return -ol[e]-(pot[G.head(e)]-pot[G.tail(e)]);   
      }     
	
      ModLengthMap(const ResGraphType& _G,
		   const LengthMap &o,  const NodeMap &p) : 
	G(_G), /*rev(_rev),*/ ol(o), pot(p){}; 
    };//ModLengthMap


  protected:
    
    //Input
    const Graph& G;
    const LengthMap& length;
    const SupplyMap& supply;//supply or demand of nodes


    //auxiliary variables

    //To store the flow
    EdgeIntMap flow; 
    //To store the potentila (dual variables)
    typename Graph::template NodeMap<Length> potential;
    //To store excess-deficit values
    SupplyMap excess;
    

    Length total_length;


  public :


    MinCostFlow(Graph& _G, LengthMap& _length, SupplyMap& _supply) : G(_G), 
      length(_length), supply(_supply), flow(_G), potential(_G){ }

    
    ///Runs the algorithm.

    ///Runs the algorithm.
    ///Returns k if there are at least k edge-disjoint paths from s to t.
    ///Otherwise it returns the number of found edge-disjoint paths from s to t.
    ///\todo May be it does make sense to be able to start with a nonzero 
    /// feasible primal-dual solution pair as well.
    int run() {

      //Resetting variables from previous runs
      total_length = 0;
      
      FOR_EACH_LOC(typename Graph::EdgeIt, e, G){
	flow.set(e,0);
      }

      //Initial value for delta
      Supply delta = 0;
      
      FOR_EACH_LOC(typename Graph::NodeIt, n, G){
	if (delta < abs(supply[e])){
	  delta = abs(supply[e]);
	}
	excess.set(n,supply[e]);
	//Initialize the copy of the Dijkstra potential to zero
	potential.set(n,0);
      }
      

      
      //We need a residual graph which is uncapacitated
      ResGraphType res_graph(G, flow);


      
      ModLengthMap mod_length(res_graph, length, potential);

      Dijkstra<ResGraphType, ModLengthMap> dijkstra(res_graph, mod_length);


      int i;
      for (i=0; i<k; ++i){
	dijkstra.run(s);
	if (!dijkstra.reached(t)){
	  //There are no k paths from s to t
	  break;
	};
	
	//We have to copy the potential
	FOR_EACH_LOC(typename ResGraphType::NodeIt, n, res_graph){
	  potential[n] += dijkstra.distMap()[n];
	}

	/*
	{

	  typename ResGraphType::NodeIt n;
	  for ( res_graph.first(n) ; res_graph.valid(n) ; res_graph.next(n) ) {
	      potential[n] += dijkstra.distMap()[n];
	  }
	}
	*/

	//Augmenting on the sortest path
	Node n=t;
	ResGraphEdge e;
	while (n!=s){
	  e = dijkstra.pred(n);
	  n = dijkstra.predNode(n);
	  res_graph.augment(e,delta);
	  //Let's update the total length
	  if (res_graph.forward(e))
	    total_length += length[e];
	  else 
	    total_length -= length[e];	    
	}

	  
      }
      

      return i;
    }




    ///This function gives back the total length of the found paths.
    ///Assumes that \c run() has been run and nothing changed since then.
    Length totalLength(){
      return total_length;
    }

    ///Returns a const reference to the EdgeMap \c flow. \pre \ref run() must
    ///be called before using this function.
    const EdgeIntMap &getFlow() const { return flow;}

  ///Returns a const reference to the NodeMap \c potential (the dual solution).
    /// \pre \ref run() must be called before using this function.
    const EdgeIntMap &getPotential() const { return potential;}

    ///This function checks, whether the given solution is optimal
    ///Running after a \c run() should return with true
    ///In this "state of the art" this only check optimality, doesn't bother with feasibility
    ///
    ///\todo Is this OK here?
    bool checkComplementarySlackness(){
      Length mod_pot;
      Length fl_e;
      FOR_EACH_LOC(typename Graph::EdgeIt, e, G){
	//C^{\Pi}_{i,j}
	mod_pot = length[e]-potential[G.head(e)]+potential[G.tail(e)];
	fl_e = flow[e];
	//	std::cout << fl_e << std::endl;
	if (0<fl_e && fl_e<capacity[e]){
	  if (mod_pot != 0)
	    return false;
	}
	else{
	  if (mod_pot > 0 && fl_e != 0)
	    return false;
	  if (mod_pot < 0 && fl_e != capacity[e])
	    return false;
	}
      }
      return true;
    }
    

  }; //class MinCostFlow

  ///@}

} //namespace hugo

#endif //HUGO_MINCOSTFLOW_H
