// -*- c++ -*-
#ifndef MAP_DEFINES_H
#define MAP_DEFINES_H

/** Creates the EdgeMapRegistry type an declare a mutable instance 
 *  named edge_maps.
 */
#define CREATE_EDGE_MAP_REGISTRY \
typedef MapRegistry<Graph, Edge, EdgeIt> EdgeMapRegistry; \
mutable EdgeMapRegistry edge_maps;

/** Creates the NodeMapRegistry type an declare a mutable instance 
 *  named node_maps.
 */
#define CREATE_NODE_MAP_REGISTRY \
typedef MapRegistry<Graph, Node, NodeIt> NodeMapRegistry; \
mutable NodeMapRegistry node_maps;

/** Creates both map registries.
 */
#define CREATE_MAP_REGISTRIES \
CREATE_NODE_MAP_REGISTRY \
CREATE_EDGE_MAP_REGISTRY

/** Creates a map a concrete factory type from a template map
 *  factory to use as node map factory.
 */
#define CREATE_NODE_MAP_FACTORY(TemplateFactory) \
typedef TemplateFactory<NodeMapRegistry> NodeMapFactory;

/** Creates a map a concrete factory type from a template map
 *  factory to use as edge map factory.
 */
#define CREATE_EDGE_MAP_FACTORY(TemplateFactory) \
typedef TemplateFactory<EdgeMapRegistry> EdgeMapFactory;

/** Creates both map factories.
 */
#define CREATE_MAP_FACTORIES(TemplateFactory) \
CREATE_NODE_MAP_FACTORY(TemplateFactory) \
CREATE_EDGE_MAP_FACTORY(TemplateFactory) 

/** Import a map from a concrete map factory. The import method is
 *  an overloading of the map type.
 *  The reason to use these macro is that the c++ does not support
 *  the template typedefs. If a future release of the c++ 
 *  supports this feature it should be fixed.
 */
#define IMPORT_NODE_MAP(Factory) \
template <typename V> \
class NodeMap : public Factory::Map<V> { \
public: \
NodeMap() {} \
NodeMap(const Graph& g) : Factory::Map<V>(&g, &(g.node_maps)) {} \
NodeMap(const Graph& g, const V& v) : Factory::Map<V>(g, g.node_maps, v) {} \
NodeMap(const NodeMap& copy) \
  : Factory::Map<V>(static_cast<const Factory::Map<V>&>(copy)) {} \
template <typename CMap> NodeMap(const CMap& copy) : Factory::Map<V>(copy) {} \
NodeMap& operator=(const NodeMap& copy) { \
  this->Factory::Map<V>::operator=(static_cast<Factory::Map<V>&>(copy)); \
  return *this; \
} \
template <typename CMap>NodeMap& operator=(const CMap& copy) { \
  this->Factory::Map<V>::operator=<CMap>(copy); \
  return *this; \
} \
};

/** Import a map from a concrete map factory. The import method is
 *  an overloading of the map type.
 *  The reason to use these macro is that the c++ does not support
 *  the template typedefs. If a future release of the c++ 
 *  supports this feature it should be fixed.
 */
#define IMPORT_EDGE_MAP(Factory) \
template <typename V> \
class EdgeMap : public Factory::Map<V> { \
public: \
EdgeMap() {} \
EdgeMap(const Graph& g) : Factory::Map<V>(g, g.edge_maps) {} \
EdgeMap(const Graph& g, const V& v) : Factory::Map<V>(g, g.node_maps, v) {} \
EdgeMap(const EdgeMap& copy) \
  : Factory::Map<V>(static_cast<Factory::Map<V>&>(copy)) {} \
template <typename CMap> EdgeMap(const CMap& copy) : Factory::Map<V>(copy) {} \
EdgeMap& operator=(const EdgeMap& copy) { \
  this->Factory::Map<V>::operator=(static_cast<Factory::Map<V>&>(copy)); \
  return *this; \
} \
template <typename CMap>EdgeMap& operator=(const CMap& copy) { \
  this->Factory::Map<V>::operator=<CMap>(copy); \
  return *this; \
} \
};

/** This macro creates both map factories and imports both maps.
 */
#define CREATE_MAPS(TemplateFactory) \
CREATE_MAP_FACTORIES(TemplateFactory) \
IMPORT_NODE_MAP(NodeMapFactory) \
IMPORT_EDGE_MAP(EdgeMapFactory)

#endif
