// -*- c++ -*-
#ifndef HUGO_MINCOSTFLOWS_H
#define HUGO_MINCOSTFLOWS_H

///\ingroup galgs
///\file
///\brief An algorithm for finding a flow of value \c k (for small values of \c k) having minimal total cost 

#include <iostream>
#include <hugo/dijkstra.h>
#include <hugo/graph_wrapper.h>
#include <hugo/maps.h>
#include <vector>
#include <for_each_macros.h>

namespace hugo {

/// \addtogroup galgs
/// @{

  ///\brief Implementation of an algorithm for finding a flow of value \c k 
  ///(for small values of \c k) having minimal total cost between 2 nodes 
  /// 
  ///
  /// The class \ref hugo::MinCostFlows "MinCostFlows" implements
  /// an algorithm for finding a flow of value \c k 
  ///(for small values of \c k) having minimal total cost  
  /// from a given source node to a given target node in an
  /// edge-weighted directed graph having nonnegative integer capacities.
  /// The range of the length (weight) function is nonnegative reals but 
  /// the range of capacity function is the set of nonnegative integers. 
  /// It is not a polinomial time algorithm for counting the minimum cost
  /// maximal flow, since it counts the minimum cost flow for every value 0..M
  /// where \c M is the value of the maximal flow.
  ///
  ///\author Attila Bernath
  template <typename Graph, typename LengthMap, typename CapacityMap>
  class MinCostFlows {

    typedef typename LengthMap::ValueType Length;

    //Warning: this should be integer type
    typedef typename CapacityMap::ValueType Capacity;
    
    typedef typename Graph::Node Node;
    typedef typename Graph::NodeIt NodeIt;
    typedef typename Graph::Edge Edge;
    typedef typename Graph::OutEdgeIt OutEdgeIt;
    typedef typename Graph::template EdgeMap<int> EdgeIntMap;

    //    typedef ConstMap<Edge,int> ConstMap;

    typedef ResGraphWrapper<const Graph,int,CapacityMap,EdgeIntMap> ResGraphType;
    typedef typename ResGraphType::Edge ResGraphEdge;

    class ModLengthMap {   
      //typedef typename ResGraphType::template NodeMap<Length> NodeMap;
      typedef typename Graph::template NodeMap<Length> NodeMap;
      const ResGraphType& G;
      //      const EdgeIntMap& rev;
      const LengthMap &ol;
      const NodeMap &pot;
    public :
      typedef typename LengthMap::KeyType KeyType;
      typedef typename LengthMap::ValueType ValueType;
	
      ValueType operator[](typename ResGraphType::Edge e) const {     
	if (G.forward(e))
	  return  ol[e]-(pot[G.head(e)]-pot[G.tail(e)]);   
	else
	  return -ol[e]-(pot[G.head(e)]-pot[G.tail(e)]);   
      }     
	
      ModLengthMap(const ResGraphType& _G,
		   const LengthMap &o,  const NodeMap &p) : 
	G(_G), /*rev(_rev),*/ ol(o), pot(p){}; 
    };//ModLengthMap


  protected:
    
    //Input
    const Graph& G;
    const LengthMap& length;
    const CapacityMap& capacity;


    //auxiliary variables

    //To store the flow
    EdgeIntMap flow; 
    //To store the potentila (dual variables)
    typename Graph::template NodeMap<Length> potential;
    
    //Container to store found paths
    //std::vector< std::vector<Edge> > paths;
    //typedef DirPath<Graph> DPath;
    //DPath paths;


    Length total_length;


  public :


    MinCostFlows(Graph& _G, LengthMap& _length, CapacityMap& _cap) : G(_G), 
      length(_length), capacity(_cap), flow(_G), potential(_G){ }

    
    ///Runs the algorithm.

    ///Runs the algorithm.
    ///Returns k if there are at least k edge-disjoint paths from s to t.
    ///Otherwise it returns the number of found edge-disjoint paths from s to t.
    ///\todo May be it does make sense to be able to start with a nonzero 
    /// feasible primal-dual solution pair as well.
    int run(Node s, Node t, int k) {

      //Resetting variables from previous runs
      total_length = 0;
      
      FOR_EACH_LOC(typename Graph::EdgeIt, e, G){
	flow.set(e,0);
      }
      
      FOR_EACH_LOC(typename Graph::NodeIt, n, G){
	//cout << potential[n]<<endl;
	potential.set(n,0);
      }
      

      
      //We need a residual graph
      ResGraphType res_graph(G, capacity, flow);

      //Initialize the copy of the Dijkstra potential to zero
      
      //typename ResGraphType::template NodeMap<Length> potential(res_graph);


      ModLengthMap mod_length(res_graph, length, potential);

      Dijkstra<ResGraphType, ModLengthMap> dijkstra(res_graph, mod_length);

      int i;
      for (i=0; i<k; ++i){
	dijkstra.run(s);
	if (!dijkstra.reached(t)){
	  //There are no k paths from s to t
	  break;
	};
	
	{
	  //We have to copy the potential
	  typename ResGraphType::NodeIt n;
	  for ( res_graph.first(n) ; res_graph.valid(n) ; res_graph.next(n) ) {
	      potential[n] += dijkstra.distMap()[n];
	  }
	}


	//Augmenting on the sortest path
	Node n=t;
	ResGraphEdge e;
	while (n!=s){
	  e = dijkstra.pred(n);
	  n = dijkstra.predNode(n);
	  res_graph.augment(e,1);
	  //Let's update the total length
	  if (res_graph.forward(e))
	    total_length += length[e];
	  else 
	    total_length -= length[e];	    
	}

	  
      }
      

      return i;
    }




    ///This function gives back the total length of the found paths.
    ///Assumes that \c run() has been run and nothing changed since then.
    Length totalLength(){
      return total_length;
    }

    ///Returns a const reference to the EdgeMap \c flow. \pre \ref run() must
    ///be called before using this function.
    const EdgeIntMap &getFlow() const { return flow;}

  ///Returns a const reference to the NodeMap \c potential (the dual solution).
    /// \pre \ref run() must be called before using this function.
    const EdgeIntMap &getPotential() const { return potential;}

    ///This function checks, whether the given solution is optimal
    ///Running after a \c run() should return with true
    ///In this "state of the art" this only check optimality, doesn't bother with feasibility
    ///
    ///\todo Is this OK here?
    bool checkComplementarySlackness(){
      Length mod_pot;
      Length fl_e;
      FOR_EACH_LOC(typename Graph::EdgeIt, e, G){
	//C^{\Pi}_{i,j}
	mod_pot = length[e]-potential[G.head(e)]+potential[G.tail(e)];
	fl_e = flow[e];
	//	std::cout << fl_e << std::endl;
	if (0<fl_e && fl_e<capacity[e]){
	  if (mod_pot != 0)
	    return false;
	}
	else{
	  if (mod_pot > 0 && fl_e != 0)
	    return false;
	  if (mod_pot < 0 && fl_e != capacity[e])
	    return false;
	}
      }
      return true;
    }
    
    /*
      ///\todo To be implemented later

    ///This function gives back the \c j-th path in argument p.
    ///Assumes that \c run() has been run and nothing changed since then.
    /// \warning It is assumed that \c p is constructed to be a path of graph \c G. If \c j is greater than the result of previous \c run, then the result here will be an empty path.
    template<typename DirPath>
    void getPath(DirPath& p, int j){
      p.clear();
      typename DirPath::Builder B(p);
      for(typename std::vector<Edge>::iterator i=paths[j].begin(); 
	  i!=paths[j].end(); ++i ){
	B.pushBack(*i);
      }

      B.commit();
    }

    */

  }; //class MinCostFlows

  ///@}

} //namespace hugo

#endif //HUGO_MINCOSTFLOW_H
