// -*- c++ -*-
#ifndef HUGO_DIMACS_H
#define HUGO_DIMACS_H

#include <iostream>
#include <string>
#include <vector>
#include <hugo/maps.h>

/// \ingroup misc
/// \file
/// \brief Dimacs file format reader.

namespace hugo {


  /// \addtogroup misc
  /// @{

  /// Dimacs min cost flow reader function.

  /// This function reads a min cost flow instance from dimacs format,
  /// i.e. from dimacs files having a line starting with \c p \c "min".
  /// At the beginning \c g is cleared by \c g.clear(). The edge
  /// capacities are written to \c capacity, \c s and \c t are set to
  /// the source and the target nodes resp. and the cost of the edges
  /// are written to \c cost.
  ///
  /// \author Marton Makai
  template<typename Graph, typename CapacityMap, typename CostMap>
  void readDimacs(std::istream& is, Graph &g, CapacityMap& capacity, 
		  typename Graph::Node &s, typename Graph::Node &t, 
		  CostMap& cost) {
    g.clear();
    typename CapacityMap::ValueType _cap;
    typename CostMap::ValueType _cost;
    char d;
    std::string problem;
    char c;
    int i, j;
    std::string str;
    int n, m; 
    typename Graph::Edge e;
    std::vector<typename Graph::Node> nodes;
    while (is>>c) {
      switch (c) {
      case 'c': //comment
	getline(is, str);
	break;
      case 'p': //problem definition
	is >> problem >> n >> m;
	getline(is, str);
	nodes.resize(n+1);
	for (int k=1; k<=n; ++k) nodes[k]=g.addNode();
	break;
      case 'n': //node definition
	if (problem=="sp") { //shortest path problem
	  is >> i;
	  getline(is, str);
	  s=nodes[i];
	}
	if (problem=="max" || problem=="min") { //((max) or (min cost)) flow problem
	  is >> i >> d;
	  getline(is, str);
	  if (d=='s') s=nodes[i];
	  if (d=='t') t=nodes[i];
	}
	break;
      case 'a':
	if ( problem == "max" || problem == "sp") {
	  is >> i >> j >> _cap;
	  getline(is, str);
	  e=g.addEdge(nodes[i], nodes[j]);
	  capacity.update();
	  capacity.set(e, _cap);
	} else {
	  if ( problem == "min" ) {
	    is >> i >> j >> _cap >> _cost;
	    getline(is, str);
	    e=g.addEdge(nodes[i], nodes[j]);
	    capacity.update();
	    capacity.set(e, _cap);
	    cost.update();
	    cost.set(e, _cost);
	  } else {
	    is >> i >> j;
	    getline(is, str);
	    g.addEdge(nodes[i], nodes[j]);
	  }
	}
	break;
      }
    }
  }


  /// Dimacs max flow reader function.

  /// This function reads a max flow instance from dimacs format,
  /// i.e. from dimacs files having a line starting with \c p \c
  /// "max".  At the beginning \c g is cleared by \c g.clear(). The
  /// edge capacities are written to \c capacity and \c s and \c t are
  /// set to the source and the target nodes.
  ///
  /// \author Marton Makai
  template<typename Graph, typename CapacityMap>
  void readDimacs(std::istream& is, Graph &g, CapacityMap& capacity, 
		  typename Graph::Node &s, typename Graph::Node &t) {
    NullMap<typename Graph::Edge, int> n;
    readDimacs(is, g, capacity, s, t, n);
  }


  /// Dimacs shortest path reader function.

  /// This function reads a shortest path instance from dimacs format,
  /// i.e. from dimacs files having a line starting with \c p \c "sp".
  /// At the beginning \c g is cleared by \c g.clear(). The edge
  /// capacities are written to \c capacity and \c s is set to the
  /// source node.
  ///
  /// \author Marton Makai
  template<typename Graph, typename CapacityMap>
  void readDimacs(std::istream& is, Graph &g, CapacityMap& capacity, 
		  typename Graph::Node &s) {
    NullMap<typename Graph::Edge, int> n;
    readDimacs(is, g, capacity, s, s, n);
  }


  /// Dimacs capacitated graph reader function.

  /// This function reads an edge capacitated graph instance from
  /// dimacs format.  At the beginning \c g is cleared by \c g.clear()
  /// and the edge capacities are written to \c capacity.
  ///
  /// \author Marton Makai
  template<typename Graph, typename CapacityMap>
  void readDimacs(std::istream& is, Graph &g, CapacityMap& capacity) {
    typename Graph::Node u;
    NullMap<typename Graph::Edge, int> n;
    readDimacs(is, g, capacity, u, u, n);
  }


  /// Dimacs plain graph reader function.

  /// This function reads a graph without any designated nodes and
  /// maps from dimacs format, i.e. from dimacs files having a line
  /// starting with \c p \c "mat".  At the beginning \c g is cleared
  /// by \c g.clear().
  ///
  /// \author Marton Makai
  template<typename Graph>
  void readDimacs(std::istream& is, Graph &g) {
    typename Graph::Node u;
    NullMap<typename Graph::Edge, int> n;
    readDimacs(is, g, n, u, u, n);
  }
  


  
  /// write matching problem
  template<typename Graph>
  void writeDimacs(std::ostream& os, const Graph &g) {
    typedef typename Graph::NodeIt NodeIt;
    typedef typename Graph::EdgeIt EdgeIt;  
    
    typename Graph::template NodeMap<int> nodes(g);

    os << "c matching problem" << std::endl;

    int i=1;
    NodeIt v;
    for(g.first(v); g.valid(v); g.next(v)) {
      nodes.set(v, i);
      ++i;
    }    
    
    os << "p mat " << g.nodeNum() << " " << g.edgeNum() << std::endl;

    EdgeIt e;
    for(g.first(e); g.valid(e); g.next(e)) {
      os << "a " << nodes[g.tail(e)] << " " << nodes[g.head(e)] << std::endl; 
    }

  }


  /// @}

} //namespace hugo

#endif //HUGO_DIMACS_H

//  template<typename Graph, typename CapacityMap>
//   void readDimacsMaxFlow(std::istream& is, Graph &g, 
// 			 typename Graph::Node &s, typename Graph::Node &t, CapacityMap& capacity) {
//     g.clear();
//     int cap;
//     char d;
//     std::string problem;
//     char c;
//     int i, j;
//     std::string str;
//     int n, m; 
//     typename Graph::Edge e;
//     std::vector<typename Graph::Node> nodes;
//     while (is>>c) {
//       switch (c) {
//       case 'c': //comment
// 	getline(is, str);
// 	break;
//       case 'p': //problem definition
// 	is >> problem >> n >> m;
// 	getline(is, str);
// 	nodes.resize(n+1);
// 	for (int k=1; k<=n; ++k) nodes[k]=g.addNode();
// 	break;
//       case 'n': //node definition
// 	if (problem=="sp") { //shortest path problem
// 	  is >> i;
// 	  getline(is, str);
// 	  s=nodes[i];
// 	}
// 	if (problem=="max") { //max flow problem
// 	  is >> i >> d;
// 	  getline(is, str);
// 	  if (d=='s') s=nodes[i];
// 	  if (d=='t') t=nodes[i];
// 	}
// 	break;
//       case 'a':
// 	is >> i >> j >> cap;
// 	getline(is, str);
// 	e=g.addEdge(nodes[i], nodes[j]);
// 	capacity.update();
// 	capacity.set(e, cap);
// 	break;
//       }
//     }
//   }
