/* -*- C++ -*-
 *
 * This file is a part of LEMON, a generic C++ optimization library
 *
 * Copyright (C) 2003-2006
 * Egervary Jeno Kombinatorikus Optimalizalasi Kutatocsoport
 * (Egervary Research Group on Combinatorial Optimization, EGRES).
 *
 * Permission to use, modify and distribute this software is granted
 * provided that this copyright notice appears in all copies. For
 * precise terms see the accompanying LICENSE file.
 *
 * This software is provided "AS IS" with no warranty of any kind,
 * express or implied, and with no claim as to its suitability for any
 * purpose.
 *
 */

#ifndef ALGOBOX_H
#define ALGOBOX_H

class MapStorage;
class MapSelector;

#include <all_include.h>
#include <libgnomecanvasmm.h>
#include <libgnomecanvasmm/polygon.h>
#include "map_value.h"

///Ancestor class of algorithm digraphical interface classes.

///It also demonstrates, how should an algorithm digraphical interface
///work. Children of this class have the same functions and attributes,
///therefore with all of them can the holder \ref AlgoWin communicate
///in the same way.
///
///IMPORTANT! In a child class only the following tasks are to do:
///
///-call \ref init function with correct parameters from correctly parametrized constructor
///
///-implement \ref build_box function
///
///-implement \ref run function
///
///because all other thing is automatically done in \ref init function!

class AlgoBox : public Gtk::VBox
{
  ///Signal emitted in case of need for list of maps.

  ///If the user has selected different tab to work on
  ///new maps are selected as well. These new maps should be
  ///provided for \ref AlgoBox. To get these maps, \ref AlgoBox
  ///emits this signal.
  sigc::signal<void, std::string> signal_maplist_need;

  ///Signal emitted in case of need for \ref NewMapWin.

  ///If user wants to create a new for an input, or output
  ///it can let \ref NewMapWin popped up from here as well.
  ///In that case will be this signal emitted.
  sigc::signal<void, std::string, bool> signal_newmapwin_need;

  ///Signal emitted when maplists are updated after tab change
  sigc::signal<void> signal_maplist_updated;


protected:
  ///Holder of tabnames.
  Gtk::ComboBoxText tabcbt;

  ///Holder of widgets, in which nodemaps can be selected to work on.
  std::vector<MapSelector *> nodemapcbts;

  ///Holder of widgets, in which arcmaps can be selected to work on.
  std::vector<MapSelector *> arcmapcbts;

  ///Maps of selected tabs.
  MapStorage * mapstorage;

public:
  ///Empty constructor called by children.
  AlgoBox(){};

  ///Constructor

  ///Calls \ref init function
  ///with the provided parameters. \ref init function
  ///is needed, because it is virtual, therefore the
  ///functions of the proper class will be called when
  ///running.
  ///\param tablist list of tabs in \ref MainWin
  AlgoBox(std::vector<std::string> tablist);

  ///Initiates \ref AlgoBox.

  ///Creates the digraphical interface for the realized algorithm, initiates variables, connects signals.
  ///
  ///List of tabs in \ref MainWin is required, but no one
  ///will be selected automatically. Every other
  ///entry field remains empty (unselected), until a \ref NoteBookTab
  ///is selected.
  ///
  ///It also have to bind all the signals to the correct place.
  ///This function is virtual, in all type of children of
  ///\ref AlgoBox the correct function willbe called.
  ///
  ///Therefore it is IMPORTANT that only \ref run and \ref build_box
  ///has to be implemented in children of \ref AlgoBox, every other
  ///thing will automatically work properly by the help of this
  ///function that must be called in constructor of child!!!
  virtual void init(std::vector<std::string>);

  ///Signal emitted, when selected tab changes, and new list of maps required.
  sigc::signal<void, std::string> signal_maplist_needed();

  ///Emitted if user wants to create a new map for inpuit or output.
  sigc::signal<void, std::string, bool> signal_newmapwin_needed(){return signal_newmapwin_need;};

  sigc::signal<void> signal_upon_maplist_updated(){return signal_maplist_updated;};

  ///Emits signal that requires list of maps for the recently selected \ref NoteBookTab.
  void emit_tab_change();

  ///Interface, through which \ref AlgoBox can be notified about tab addition, deletion in \ref MainWin

  ///\param tl list
  ///of new tab state.
  void update_tablist( std::vector< std::string > tl );

  ///Interface, through which \ref AlgoBox can get the maps of the recently selected \ref NoteBookTab

  ///\param ms the maps
  ///of the recently selected \ref NoteBookTab
  void update_maplist( MapStorage * ms);

  ///Interface, through which \ref AlgoBox can be notified about nodemap addition.

  ///If new map was added to \ref MapStorage of currently selected \ref NoteBookTab
  ///a signal is emitted by it. This signal is connected to this function, so \ref MapSelector s
  ///in \ref nodemapcbts can be notified, and those can registrate the new map. (\ref MapSelector::append_text)
  void nodemaplist_changed(std::string, MapValue::Type);

  ///Interface, through which \ref AlgoBox can be notified about arcmap addition.

  ///If new map was added to \ref MapStorage of currently selected \ref NoteBookTab
  ///a signal is emitted by it. This signal is connected to this function, so \ref MapSelector s
  ///in \ref arcmapcbts can be notified, and those can registrate the new map. (\ref MapSelector::append_text)
  void arcmaplist_changed(std::string, MapValue::Type);

  ///Aid function to provide data for a given entry.

  ///At the moment it is only used for updating info
  ///in \ref tabcbt. It clears it first, after that
  ///inserts the data got from caller, and if there
  ///was previously selected item it switches entry
  ///to that.
  ///\param tl list of entries (at the moment tabs in \ref MainWin)
  ///\param cbt the entry to update (at the moment only \ref tabcbt)
  void update_cbt( std::vector< std::string > tl, Gtk::ComboBoxText & cbt);

  ///Runs the ralized algorithm.

  ///Prepare the data for it
  ///and after that postprocess it if necessary.
  ///This is only a demo here, but in children it
  ///runs the algorithm really.
  virtual void run();

  ///Creates the layout of the \ref AlgoBox

  ///Place all the entries
  ///required. Run and close button is not
  ///its responsibility!
  virtual void build_box();

  ///Emits \ref signal_newmapwin_need if user wants to create new input or output map.

  ///Called in case of pressing \ref MapSelector::newbut.
  ///\param itisarc arc or nodemap is required.
  virtual void emit_new_map_signal(bool itisarc);

  ///Aid function to make addition of \ref MapSelector easy in \ref build_box.

  ///\param label label to show in \ref MapSelector
  ///\param itisarc whether arc or nodemaps stored in \ref MapSelector
  void addMapSelector(std::string label, bool itisarc, MapType type = ALL);
};
#endif //ALGOBOX_H
