/* -*- C++ -*-
 *
 * This file is a part of LEMON, a generic C++ optimization library
 *
 * Copyright (C) 2003-2006
 * Egervary Jeno Kombinatorikus Optimalizalasi Kutatocsoport
 * (Egervary Research Group on Combinatorial Optimization, EGRES).
 *
 * Permission to use, modify and distribute this software is granted
 * provided that this copyright notice appears in all copies. For
 * precise terms see the accompanying LICENSE file.
 *
 * This software is provided "AS IS" with no warranty of any kind,
 * express or implied, and with no claim as to its suitability for any
 * purpose.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "main_win.h"
#include "guipixbufs.h"
#include "save_details_dialog.h"
#include "background_chooser_dialog.h"

#include <mapstorage.h>
#include <graph_displayer_canvas.h>
#include <algowin.h>
#include <new_map_win.h>
#include <nbtab.h>

#include "i18n.h"

MainWin::MainWin()
{
  set_title ("no file");
  set_default_size(WIN_WIDTH,WIN_HEIGHT);
  //add(vbox);
  add(table);

  // custom icons for the toolbar
  Glib::RefPtr<Gtk::IconFactory> p_icon_factory = Gtk::IconFactory::create();
 
  Glib::RefPtr<Gdk::Pixbuf> p_move_pixbuf = Gdk::Pixbuf::create_from_inline(
      2328, gui_icons_move);
  Glib::RefPtr<Gdk::Pixbuf> p_addnode_pixbuf = Gdk::Pixbuf::create_from_inline(
      2328, gui_icons_addnode);
  Glib::RefPtr<Gdk::Pixbuf> p_addlink_pixbuf = Gdk::Pixbuf::create_from_inline(
      2328, gui_icons_addlink);
  Glib::RefPtr<Gdk::Pixbuf> p_delete_pixbuf = Gdk::Pixbuf::create_from_inline(
      2328, gui_icons_delete);
  Glib::RefPtr<Gdk::Pixbuf> p_editlink_pixbuf = Gdk::Pixbuf::create_from_inline(
      2328, gui_icons_editlink);
  Glib::RefPtr<Gdk::Pixbuf> p_newmap_pixbuf = Gdk::Pixbuf::create_from_inline(
      2328, gui_icons_newmap);
  Glib::RefPtr<Gdk::Pixbuf> p_eps_pixbuf = Gdk::Pixbuf::create_from_inline(
      2328, gui_icons_eps);

  Gtk::IconSource move_icon_source;
  move_icon_source.set_pixbuf(p_move_pixbuf);
  Gtk::IconSet move_icon_set;
  move_icon_set.add_source(move_icon_source);
  p_icon_factory->add(Gtk::StockID("gd-move"), move_icon_set);

  Gtk::IconSource addnode_icon_source;
  addnode_icon_source.set_pixbuf(p_addnode_pixbuf);
  Gtk::IconSet addnode_icon_set;
  addnode_icon_set.add_source(addnode_icon_source);
  p_icon_factory->add(Gtk::StockID("gd-addnode"), addnode_icon_set);

  Gtk::IconSource addlink_icon_source;
  addlink_icon_source.set_pixbuf(p_addlink_pixbuf);
  Gtk::IconSet addlink_icon_set;
  addlink_icon_set.add_source(addlink_icon_source);
  p_icon_factory->add(Gtk::StockID("gd-addlink"), addlink_icon_set);

  Gtk::IconSource delete_icon_source;
  delete_icon_source.set_pixbuf(p_delete_pixbuf);
  Gtk::IconSet delete_icon_set;
  delete_icon_set.add_source(delete_icon_source);
  p_icon_factory->add(Gtk::StockID("gd-delete"), delete_icon_set);

  Gtk::IconSource editlink_icon_source;
  editlink_icon_source.set_pixbuf(p_editlink_pixbuf);
  Gtk::IconSet editlink_icon_set;
  editlink_icon_set.add_source(editlink_icon_source);
  p_icon_factory->add(Gtk::StockID("gd-editlink"), editlink_icon_set);

  Gtk::IconSource newmap_icon_source;
  newmap_icon_source.set_pixbuf(p_newmap_pixbuf);
  Gtk::IconSet newmap_icon_set;
  newmap_icon_set.add_source(newmap_icon_source);
  p_icon_factory->add(Gtk::StockID("gd-newmap"), newmap_icon_set);

  Gtk::IconSource eps_icon_source;
  eps_icon_source.set_pixbuf(p_eps_pixbuf);
  Gtk::IconSet eps_icon_set;
  eps_icon_set.add_source(eps_icon_source);
  p_icon_factory->add(Gtk::StockID("gd-eps"), eps_icon_set);

  p_icon_factory->add_default();
  
  ag=Gtk::ActionGroup::create();

  ag->add( Gtk::Action::create("FileMenu", _("_File")) );
  ag->add( Gtk::Action::create("FileNew", Gtk::Stock::NEW),
      sigc::mem_fun(*this, &MainWin::newTab));
  ag->add( Gtk::Action::create("FileOpen", Gtk::Stock::OPEN),
      sigc::mem_fun(*this, &MainWin::openFile));
  ag->add( Gtk::Action::create("FileClearTab", _("Clear Tab")),
      sigc::mem_fun(*this, &MainWin::newFile));
  ag->add( Gtk::Action::create("FileSave", Gtk::Stock::SAVE),
      sigc::mem_fun(*this, &MainWin::saveFile));
  ag->add( Gtk::Action::create("FileSaveAs", Gtk::Stock::SAVE_AS),
      sigc::mem_fun(*this, &MainWin::saveFileAs));
  ag->add( Gtk::Action::create("SaveDetails", _("Save _Details...")),
	   sigc::mem_fun(*this, &MainWin::createSaveDetailsDialog));
  ag->add( Gtk::Action::create("Close", Gtk::Stock::CLOSE),
      sigc::mem_fun(*this, &MainWin::closeTab));
  ag->add( Gtk::Action::create("Quit", Gtk::Stock::QUIT),
      sigc::mem_fun(*this, &MainWin::hide));

  ag->add( Gtk::Action::create("ViewMenu", _("_View")) );
  ag->add( Gtk::Action::create("ViewZoomIn", Gtk::Stock::ZOOM_IN),
      sigc::mem_fun(*this, &MainWin::zoomIn));
  ag->add( Gtk::Action::create("ViewZoomOut", Gtk::Stock::ZOOM_OUT),
      sigc::mem_fun(*this, &MainWin::zoomOut));
  ag->add( Gtk::Action::create("ViewZoomFit", Gtk::Stock::ZOOM_FIT),
      sigc::mem_fun(*this, &MainWin::zoomFit));
  ag->add( Gtk::Action::create("ViewZoom100", Gtk::Stock::ZOOM_100),
      sigc::mem_fun(*this, &MainWin::zoom100));
  ag->add( Gtk::Action::create("SetBackground", _("Set Background...")),
	    sigc::mem_fun(*this, &MainWin::createBackgroundChooser));
  
  ag->add( Gtk::Action::create("ShowMenu", _("_Show")) );
  ag->add( Gtk::Action::create("ShowMaps", _("_Maps")),
	   sigc::mem_fun(*this, &MainWin::createMapWin));
  ag->add( Gtk::Action::create("ShowDesign", _("_Design")),
	   sigc::mem_fun(*this, &MainWin::createDesignWin));

  ag->add( Gtk::Action::create("AlgoMenu", _("_Algorithms")) );
  ag->add( Gtk::Action::create("AlgoGeneral", _("_General")),
	   sigc::bind( sigc::mem_fun ( *this, &MainWin::createAlgoWin ), 0) );
  ag->add( Gtk::Action::create("AlgoKruskal", _("_Kruskal")),
	   sigc::bind( sigc::mem_fun ( *this, &MainWin::createAlgoWin ), 1) );
  ag->add( Gtk::Action::create("AlgoDijkstra", _("_Dijkstra")),
	   sigc::bind( sigc::mem_fun ( *this, &MainWin::createAlgoWin ), 2) );
  ag->add( Gtk::Action::create("AlgoSuurballe", _("_Suurballe")),
	   sigc::bind( sigc::mem_fun ( *this, &MainWin::createAlgoWin ), 3) );

  Gtk::RadioAction::Group tool_group;
  ag->add( Gtk::RadioAction::create(tool_group, "MoveItem", Gtk::StockID("gd-move"), _("Move")),
      sigc::bind( sigc::mem_fun ( *this, &MainWin::changeEditorialTool ), 0) );
  ag->add( Gtk::RadioAction::create(tool_group, "CreateNode", Gtk::StockID("gd-addnode"), _("Create node")),
      sigc::bind( sigc::mem_fun ( *this, &MainWin::changeEditorialTool ), 1) );
  ag->add( Gtk::RadioAction::create(tool_group, "CreateArc", Gtk::StockID("gd-addlink"), _("Create arc")),
      sigc::bind( sigc::mem_fun ( *this, &MainWin::changeEditorialTool ), 2) );
  ag->add( Gtk::RadioAction::create(tool_group, "EraseItem", Gtk::StockID("gd-delete"), _("Delete")),
      sigc::bind( sigc::mem_fun ( *this, &MainWin::changeEditorialTool ), 3) );

  ag->add( Gtk::RadioAction::create(tool_group, "EditArcMap", Gtk::StockID("gd-editlink"), _("Edit arc map")),
      sigc::bind( sigc::mem_fun ( *this, &MainWin::changeEditorialTool ), 4) );

  ag->add( Gtk::Action::create("AddMap", Gtk::StockID("gd-newmap"), "New map"),
      sigc::mem_fun ( *this , &MainWin::createNewMapWin ) );

  ag->add( Gtk::Action::create("DesignDigraph", Gtk::Stock::REFRESH),
      sigc::mem_fun ( *this , &MainWin::reDesignDigraph ) );

  ag->add( Gtk::Action::create("Eps", Gtk::StockID("gd-eps"), "Export to EPS"),
      sigc::mem_fun ( *this , &MainWin::exportToEPS ) );

  uim=Gtk::UIManager::create();
  uim->insert_action_group(ag);
  add_accel_group(uim->get_accel_group());

  try
  {

    Glib::ustring ui_info =
      "<ui>"
      "  <menubar name='MenuBar'>"
      "    <menu action='FileMenu'>"
      "      <menuitem action='FileNew'/>"
      "      <menuitem action='FileOpen'/>"
      "      <menuitem action='FileClearTab'/>"
      "      <menuitem action='FileSave'/>"
      "      <menuitem action='FileSaveAs'/>"
      "      <menuitem action='SaveDetails'/>"
      "      <menuitem action='Close'/>"
      "      <menuitem action='Quit'/>"
      "    </menu>"
      "    <menu action='ViewMenu'>"
      "      <menuitem action='ViewZoomIn' />"
      "      <menuitem action='ViewZoomOut' />"
      "      <menuitem action='ViewZoom100' />"
      "      <menuitem action='ViewZoomFit' />"
      "      <menuitem action='SetBackground' />"
      "    </menu>"
      "    <menu action='ShowMenu'>"
      "      <menuitem action='ShowMaps'/>"
      "      <menuitem action='ShowDesign'/>"
      "    </menu>"
      "    <menu action='AlgoMenu'>"
      "      <menuitem action='AlgoGeneral'/>"
      "      <menuitem action='AlgoKruskal'/>"
      "      <menuitem action='AlgoDijkstra'/>"
      "      <menuitem action='AlgoSuurballe'/>"
      "    </menu>"
      "  </menubar>"
      "  <toolbar name='ToolBar'>"
      "    <toolitem action='FileNew' />"
      "    <toolitem action='FileOpen' />"
      "    <toolitem action='FileSave' />"
      "    <toolitem action='Close' />"
      "    <separator />"
      "    <toolitem action='ViewZoomIn' />"
      "    <toolitem action='ViewZoomOut' />"
      "    <toolitem action='ViewZoom100' />"
      "    <toolitem action='ViewZoomFit' />"
      "    <separator />"
      "    <toolitem action='MoveItem' />"
      "    <toolitem action='CreateNode' />"
      "    <toolitem action='CreateArc' />"
      "    <toolitem action='EraseItem' />"
      "    <toolitem action='EditArcMap' />"
      "    <separator />"
      "    <toolitem action='AddMap' />"
      "    <toolitem action='DesignDigraph' />"
      "    <toolitem action='Eps' />"
      "  </toolbar>"
      "</ui>";

    uim->add_ui_from_string(ui_info);

  }
  catch(const Glib::Error& ex)
  {
    std::cerr << "building menus failed: " <<  ex.what();
  }

  Gtk::Widget* menubar = uim->get_widget("/MenuBar");
  if (menubar){
    //vbox.pack_start(*menubar, Gtk::PACK_SHRINK);
    table.attach(*menubar, 0, 1, 0, 1, Gtk::EXPAND|Gtk::FILL, Gtk::SHRINK);
  }

  Gtk::Widget* toolbar = uim->get_widget("/ToolBar");
  if (toolbar)
  {
    static_cast<Gtk::Toolbar*>(toolbar)->set_toolbar_style(Gtk::TOOLBAR_ICONS);
    static_cast<Gtk::Toolbar*>(toolbar)->set_tooltips(true);
    //hbox.pack_start(*toolbar, Gtk::PACK_EXPAND_WIDGET);

    table.attach(*toolbar, 0, 1, 1, 2, Gtk::EXPAND|Gtk::FILL, Gtk::SHRINK);

  }
 
  table2.set_row_spacings(10);
  table2.set_col_spacings(5);

  auto_scale = Gtk::manage(new Gtk::CheckButton("Autoscale"));
  auto_scale->set_active(false);
  auto_scale->signal_toggled().connect(sigc::mem_fun(*this, &MainWin::nodeViewChanged));
  table2.attach(*auto_scale, 0,2,0,1);

  Gtk::Label * width_label=Gtk::manage(new Gtk::Label("Arc Width:"));
  table2.attach(*width_label, 0,1,1,2);
  
  Gtk::Adjustment * adjustment_width=Gtk::manage(new Gtk::Adjustment(20, 1, 200, 5, 10));
  
  arc_width = Gtk::manage(new Gtk::SpinButton(*adjustment_width, 5,0));
  arc_width->signal_value_changed().connect(sigc::mem_fun(*this, &MainWin::nodeViewChanged));
  table2.attach(*arc_width, 1,2,1,2);

  Gtk::Label * radius_label= Gtk::manage(new Gtk::Label("Node Radius:"));
  table2.attach(*radius_label, 2,3,1,2);
  
  Gtk::Adjustment * adjustment_radius=Gtk::manage(new Gtk::Adjustment(20, 0, 500, 5, 10));

  radius_size = Gtk::manage(new Gtk::SpinButton(*adjustment_radius, 5,0));
  radius_size->signal_value_changed().connect(sigc::mem_fun(*this, &MainWin::nodeViewChanged));
  table2.attach(*radius_size, 3,4,1,2);

  zoom_track = Gtk::manage(new Gtk::CheckButton("Zoom tracking"));
  zoom_track->set_active(false);
  zoom_track->signal_toggled().connect(sigc::mem_fun(*this, &MainWin::nodeViewChanged));
  table2.attach(*zoom_track, 2,4,0,1);


  table.attach(table2, 1, 2, 0, 2, Gtk::SHRINK, Gtk::SHRINK);

  tooltips=Gtk::manage(new Gtk::Tooltips());
  if(tooltips)
    {
      tooltips->set_tip(*zoom_track, "If on, arc widths and node radiuses are constant, independent from zooming");
      tooltips->set_tip(*auto_scale, "If on, glemon automatically determines the size of arcs and nodes");
      tooltips->set_tip(*radius_size, "Sets maximum node radius, if auto-scale is off");
      tooltips->set_tip(*arc_width, "Sets maximum arc width, if auto-scale is off");

      static_cast<Gtk::ToolItem*>(uim->get_widget("/ToolBar/FileNew"))->set_tooltip(*tooltips, "Inserts new tab");
      static_cast<Gtk::ToolItem*>(uim->get_widget("/ToolBar/FileOpen"))->set_tooltip(*tooltips, "Lets you open a file");
      static_cast<Gtk::ToolItem*>(uim->get_widget("/ToolBar/FileSave"))->set_tooltip(*tooltips, "Saves the digraph on the active tab");
      static_cast<Gtk::ToolItem*>(uim->get_widget("/ToolBar/Close"))->set_tooltip(*tooltips, "Closes the active tab");
      static_cast<Gtk::ToolItem*>(uim->get_widget("/ToolBar/ViewZoomIn"))->set_tooltip(*tooltips, "Zoom in the digraph");
      static_cast<Gtk::ToolItem*>(uim->get_widget("/ToolBar/ViewZoomOut"))->set_tooltip(*tooltips, "Zoom out the digraph");
      static_cast<Gtk::ToolItem*>(uim->get_widget("/ToolBar/ViewZoom100"))->set_tooltip(*tooltips, "Shows actual size of digraph");
      static_cast<Gtk::ToolItem*>(uim->get_widget("/ToolBar/ViewZoomFit"))->set_tooltip(*tooltips, "Fits digraph into window");
      static_cast<Gtk::ToolItem*>(uim->get_widget("/ToolBar/MoveItem"))->set_tooltip(*tooltips, "Moves the clicked item (arc/node)");
      static_cast<Gtk::ToolItem*>(uim->get_widget("/ToolBar/CreateNode"))->set_tooltip(*tooltips, "Adds new node");
      static_cast<Gtk::ToolItem*>(uim->get_widget("/ToolBar/CreateArc"))->set_tooltip(*tooltips, "Lets you create new arc");
      static_cast<Gtk::ToolItem*>(uim->get_widget("/ToolBar/EraseItem"))->set_tooltip(*tooltips, "Erases the clicked item (arc/node)");
      static_cast<Gtk::ToolItem*>(uim->get_widget("/ToolBar/EditArcMap"))->set_tooltip(*tooltips, "Lets you edit the values written on the items");
      static_cast<Gtk::ToolItem*>(uim->get_widget("/ToolBar/AddMap"))->set_tooltip(*tooltips, "Adds arc/nodemap");
      static_cast<Gtk::ToolItem*>(uim->get_widget("/ToolBar/DesignDigraph"))->set_tooltip(*tooltips, "Redesigns your digraph, supposing elastic arcs and propulsation of nodes.");

      tooltips->enable();
    }

  active_tab=-1;
  notebook.signal_switch_page().connect(sigc::mem_fun(*this, &MainWin::onChangeTab));

  active_tool = MOVE;

  table.attach(notebook,0,2,2,3, Gtk::EXPAND|Gtk::FILL, Gtk::EXPAND|Gtk::FILL);
  
  show_all_children();
}

MainWin::~MainWin()
{
}

void MainWin::set_tabtitle(std::string name)
{
  if(strinst.find(name)==strinst.end())
    {
      tabnames[active_tab]=name;
      strinst[name]=1;
    }
  else
    {
      strinst[name]++;
      std::ostringstream o;
      o << strinst[name];
      tabnames[active_tab]=name+" - "+o.str();
    }
  set_title(tabnames[active_tab] + " - " + prog_name);
  notebook.set_tab_label_text((Widget&)*(tabs[active_tab]), tabnames[active_tab]);
  updateAlgoWinTabs();
}

void MainWin::readFile(const std::string & filename)
{
  newTab();
  tabs[active_tab]->readFile(filename);
}

void MainWin::newTab()
{
  int size=tabs.size();
  tabs.resize(size+1);
  tabnames.resize(size+1);
  active_tab=size;
  tabs[active_tab]=Gtk::manage(new NoteBookTab());
  tabs[active_tab]->signal_title_ch().connect(sigc::mem_fun(*this, &MainWin::set_tabtitle));
  tabs[active_tab]->signal_newmap_needed().connect(sigc::mem_fun(*this, &MainWin::createNewMapWinAfterSignal));
  tabs[active_tab]->gd_canvas->changeEditorialTool(active_tool);
  notebook.append_page((Gtk::Widget&)(*(tabs[active_tab])));
  notebook.set_current_page(size);
  set_tabtitle(_("unsaved file"));
  updateAlgoWinTabs();
}

void MainWin::closeTab()
{
  if(active_tab!=-1)
    {
      if (tabs[active_tab]->mapstorage->getModified())
	{
	  Gtk::MessageDialog mdialog(_("<b>Save changes before closing?</b>"), true, 
				     Gtk::MESSAGE_WARNING, Gtk::BUTTONS_NONE);
	  mdialog.add_button(_("Close file _without Saving"), Gtk::RESPONSE_REJECT);
	  mdialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
	  mdialog.add_button(Gtk::Stock::SAVE, Gtk::RESPONSE_ACCEPT);
	  switch (mdialog.run())
	    {
	    case Gtk::RESPONSE_CANCEL:
	      return;
	    case Gtk::RESPONSE_REJECT:
	      break;
	    case Gtk::RESPONSE_ACCEPT:
	      tabs[active_tab]->saveFile();
	      break;
	    }
	}
      //tabs vector will be decreased with the deleted value
      int size=tabs.size();
      if(size>1)
	{
	  for(int i=active_tab+1;i<size;i++)
	    {
	      tabnames[i-1]=tabnames[i];
	      tabs[i-1]=tabs[i];
	    }
	}
      //if size==1 resize will delete the only element
      tabs.resize(size-1);
      tabnames.resize(size-1);

      int old_active_tab=active_tab;
      notebook.remove_page(active_tab);

      //If the first tab was active, upon delete notebook
      //will first switch one tab upper and not lower like
      //in the case, when not the first tab was active.
      //But after deletion it will become the first tab,
      //and this should be registrated in tabs vector,
      //as well.
      if((old_active_tab==0)&&(size!=1))
	{
	  onChangeTab(NULL,0);
	}

      //if this was the last page in notebook, there is
      //no active_tab now
      if(size==1)
	{
	  active_tab=-1;
	}

      updateAlgoWinTabs();
    }
}

void MainWin::onChangeTab(GtkNotebookPage* page, guint page_num)
{
  page=page;
  active_tab=page_num;
  tabs[active_tab]->gd_canvas->changeEditorialTool(active_tool);
  set_title(tabnames[active_tab]);
  bool autoscale;
  bool zoomtrack;
  double width;
  double radius;
  tabs[active_tab]->getView(autoscale, zoomtrack, width, radius);
  arc_width->set_value(width);
  radius_size->set_value(radius);
  zoom_track->set_active(zoomtrack);
  auto_scale->set_active(autoscale);

}

void MainWin::newFile()
{
  if(active_tab!=-1)
    {
      tabs[active_tab]->newFile();
    }
}
 
void MainWin::openFile()
{
  if(active_tab==-1)
    {
      newTab();
    }
  tabs[active_tab]->openFile();
}
 
void MainWin::saveFile()
{
  if(active_tab!=-1)
    {
      tabs[active_tab]->saveFile();
    }
}
 
void MainWin::saveFileAs()
{
  if(active_tab!=-1)
    {
      tabs[active_tab]->saveFileAs();
    }
}
 
void MainWin::close()
{
  if(active_tab!=-1)
    {
      tabs[active_tab]->close();
    }
}

void MainWin::zoomIn()
{
  if(active_tab!=-1)
    {
      tabs[active_tab]->gd_canvas->zoomIn();
    }
}

void MainWin::zoomOut()
{
  if(active_tab!=-1)
    {
      tabs[active_tab]->gd_canvas->zoomOut();
    }
}

void MainWin::zoomFit()
{
  if(active_tab!=-1)
    {
      tabs[active_tab]->gd_canvas->zoomFit();
    }
}

void MainWin::zoom100()
{
  if(active_tab!=-1)
    {
      tabs[active_tab]->gd_canvas->zoom100();
    }
}

void MainWin::createMapWin()
{
  if(active_tab!=-1)
    {
      tabs[active_tab]->createMapWin(tabnames[active_tab]);
    }
}

void MainWin::createDesignWin()
{
  if(active_tab!=-1)
    {
      tabs[active_tab]->createDesignWin(tabnames[active_tab]);
    }
}

void MainWin::createAlgoWin(int algoid)
{
  AlgoWin * aw=Gtk::manage(new AlgoWin(algoid, tabnames));
  aw->signal_closing().connect(sigc::mem_fun(*this, &MainWin::deRegisterAlgoWin));
  aw->signal_maplist_needed().connect(sigc::mem_fun(*this, &MainWin::updateAlgoWinMaps));
  aw->signal_newmapwin_needed().connect(sigc::mem_fun(*this, &MainWin::createNewMapWinTabString));
  aws.insert(aw);
  aw->show();
}

void MainWin::updateAlgoWinTabs()
{
  std::set< AlgoWin* >::iterator awsi=aws.begin();
  for(;awsi!=aws.end();awsi++)
    {
      (*awsi)->update_tablist(tabnames);
    }
}

void MainWin::updateAlgoWinMaps(AlgoWin * awp, std::string tabname)
{
  int i=0;
  for(;(i<(int)tabnames.size())&&(tabnames[i]!=tabname);i++)
    {
    }
  awp->update_maplist(tabs[i]->mapstorage);
}

void MainWin::deRegisterAlgoWin(AlgoWin * awp)
{
  aws.erase(awp);
}

void MainWin::changeEditorialTool(int tool)
{
  active_tool=tool;
  if(active_tab!=-1)
    {
      tabs[active_tab]->gd_canvas->changeEditorialTool(tool);
    }
}

void MainWin::createNewMapWin()
{
  if(active_tab!=-1)
    {
      NewMapWin * nmw=Gtk::manage(new NewMapWin(_("Create New Map - ")+tabnames[active_tab], *(tabs[active_tab])));
      nmw->show();
    }
}

void MainWin::createNewMapWinTabString(std::string tabname, bool itisarc)
{
  int i=0;
  for(;((i<(int)tabnames.size())&&(tabnames[i]!=tabname));i++)
    {
    }
  createNewMapWinAfterSignal(tabs[i], itisarc);
}

void MainWin::createNewMapWinAfterSignal(NoteBookTab * nbt, bool itisarc)
{
  std::vector<NoteBookTab*>::iterator nbti=tabs.begin();
  int i=0;
  for(;nbti!=tabs.end();nbti++)
    {
      if(*nbti!=nbt)
	{
	  i++;
	}
      else
	{
	  continue;
	}
    }
  NewMapWin * nmw=Gtk::manage(new NewMapWin(_("Create New Map - ")+tabnames[i], *nbt, itisarc, false));
  nmw->run();
}


void MainWin::nodeViewChanged()
{
  double width=arc_width->get_value();
  double radius=radius_size->get_value();
  bool zoomtrack=zoom_track->get_active();
  bool autoscale=auto_scale->get_active();
  tabs[active_tab]->setView(autoscale, zoomtrack, width, radius);
}

void MainWin::reDesignDigraph()
{
  tabs[active_tab]->reDesignDigraph();
}

void MainWin::createSaveDetailsDialog()
{
  SaveDetailsDialog dialog(tabs[active_tab]->mapstorage);
  dialog.run();
}

void MainWin::exportToEPS()
{
  if(active_tab!=-1)
    {
      tabs[active_tab]->createExportToEPSWin(tabnames[active_tab]);
    }
}

void MainWin::createBackgroundChooser()
{
  BackgroundChooserDialog dialog(tabs[active_tab]->mapstorage);
  dialog.run();
}
