/* -*- C++ -*-
 *
 * This file is a part of LEMON, a generic C++ optimization library
 *
 * Copyright (C) 2003-2006
 * Egervary Jeno Kombinatorikus Optimalizalasi Kutatocsoport
 * (Egervary Research Group on Combinatorial Optimization, EGRES).
 *
 * Permission to use, modify and distribute this software is granted
 * provided that this copyright notice appears in all copies. For
 * precise terms see the accompanying LICENSE file.
 *
 * This software is provided "AS IS" with no warranty of any kind,
 * express or implied, and with no claim as to its suitability for any
 * purpose.
 *
 */

#ifndef MAPSELECTOR_H
#define MAPSELECTOR_H

#include <all_include.h>
#include <libgnomecanvasmm.h>
#include <libgnomecanvasmm/polygon.h>
#include "map_value.h"

///A widget by which node and arcmaps can be selected, deselected and created.

///During the usage of \ref glemon we have to select
///maps several times. We also need some aid-function
///like new map creation and deselecting previously
///selected map. Instead of writing a the mapselection
///at all occurences we can use this widget by connecting
///its signals to the correct place.
class MapSelector : public Gtk::HBox
{
 protected:
  ///This signal indicates that the selection has been changed by user.
  sigc::signal<void, std::string> signal_cbt;

  ///Signal that indicates that user wants to create a new map.
  sigc::signal<void, bool> signal_newmapwin;

  ///If this is true, beyond the mapnames a 'Default' selection is available as well.

  ///For example \ref MapWin needs 'Default' option as well. In this case no map
  ///will be visualized by the appropriate property.
  ///But \ref AlgoWin do not need 'Default' option, because if no map is selected,
  ///no algorithm can be run.
  ///Its value is got and set in contructor.
  bool def;

  ///Are the names of arcmaps or nodemaps stored here.
  bool itisarc;

  ///Shows whether 'Default' option is selected or not.
  bool default_state;

  ///It is true when the new button had been pressed but the new map has not been registrated yet.

  ///Before signal of \ref NewMapWin request is emitted by the \ref MapSelector
  ///this variable is set to true. When the new map
  ///is done, it will be registrated in all existing \ref MapSelector 
  ///by \ref append_text function. That function checks
  ///whether this variable is true. If it is true that means
  ///that this \ref MapSelector has requested \ref NewMapWin.
  ///Therefore it set itself to the recently created map.
  ///After that \ref set_new_map is set again false, not to
  ///set maps active if \ref MapSelector piece is not the requester.
  bool set_new_map;

  ///The widget that holds the names of maps.

  ///It can be rolled down
  ///Names in it are selectable.
  Gtk::ComboBoxText cbt;

  std::vector<std::string> cbt_content;

  ///New button.

  ///By pressing it
  ///\ref NewMapWin wilol pop-up
  Gtk::Button newbut;

  ///Reset button.

  ///If pressed \ref cbt will
  ///set to 'Default' option.
  ///
  ///It is visible only if \ref def is true.
  Gtk::Button defbut;

  ///Shows purpose of \ref MapSelector piece.
  Gtk::Label label;

  /// Which types of maps (integer, string, ...) to display.
  MapType map_type;

 public:

  ///Constructor of \ref MapSelector

  ///Creates the layout and binds signal to the correct place.
  ///\param mapstorage Pointer to the \ref MapStorage to get the map list from.
  ///\param act preselected option
  ///\param purpose text of label indicating purpose of \ref MapStorage
  ///\param itisarc do \ref MapSelector contains arcmap names or nodemapnames.
  ///\param def do we need 'Default' option. See \ref def.
  ///\param type Specifies which types of maps to display.
  MapSelector(std::vector<std::string> n_ml,
      std::vector<std::string> s_ml, std::string act, std::string labeltext,
      bool arc, bool d = true, MapType type = ALL);

  ///Returns signal emitted if the user has changed the selection. (\ref signal_cbt)
  sigc::signal<void, std::string> signal_cbt_ch();

  ///Returns signal emitted if the user has pressed New button (\ref newbut) (\ref signal_newmapwin)
  sigc::signal<void, bool> signal_newmapwin_needed();

  ///Maintain \ref cbt.

  ///Fills in \ref cbt with names, taking
  ///into account that the previously selected option
  ///has to be set back after the operation.
  void update_list(std::vector<std::string> n_ml,
      std::vector<std::string> s_ml);

  ///Handles changement in \ref cbt.

  ///In default case it emits a signal with the selected option.
  ///But if 'Default' option is selected, it resets the \ref MapSelector
  virtual void comboChanged();

  ///Requests a \ref NewMapWin

  ///See \ref set_new_map.
  ///First it sets \ref set_new_map true to be identified
  ///at registration of new map that
  ///it has sent the \ref signal_newmapwin, therefore it
  ///has to set \ref cbt to that option.
  virtual void new_but_pressed();

  ///If called, 'Default' option is selected, that means unselection of any maps.

  ///Practically this means that if this is called,
  ///properties of digraph will set to default state.
  ///The function achieves this by emitting appropriately
  ///parametrized signal_cbt.
  virtual void reset();

  ///Returns the currently selected option.
  Glib::ustring get_active_text();

  ///Sets the parameter active in \ref cbt.

  ///\param new_value the
  ///new value to be set in \ref cbt.
  void set_active_text(Glib::ustring new_value);

  ///Sets the parameter active in \ref cbt.
  ///\param index the
  ///index of row to be set in \ref cbt.
  void set_active(int index){cbt.set_active(index);};

  ///Clear all options from \ref cbt.
  void clear(){cbt.clear();};

  ///Appends a new option to the existing ones in \ref cbt.

  ///If \ref set_new_map is true, the
  ///\ref MapSelector has requested the opened \ref NewMapWin,
  ///from that the option to append is coming. In this case
  ///this function  will set \ref cbt to the new option.
  ///\param new_option new option to append
  void append_text(Glib::ustring new_option, MapValue::Type);
};
#endif //MAPSELECTOR_H
