/* -*- C++ -*-
 *
 * This file is a part of LEMON, a generic C++ optimization library
 *
 * Copyright (C) 2003-2006
 * Egervary Jeno Kombinatorikus Optimalizalasi Kutatocsoport
 * (Egervary Research Group on Combinatorial Optimization, EGRES).
 *
 * Permission to use, modify and distribute this software is granted
 * provided that this copyright notice appears in all copies. For
 * precise terms see the accompanying LICENSE file.
 *
 * This software is provided "AS IS" with no warranty of any kind,
 * express or implied, and with no claim as to its suitability for any
 * purpose.
 *
 */

#ifndef NBTAB_H
#define NBTAB_H

class MapStorage;
class EpsWin;
class MapWin;
class DesignWin;
class DigraphDisplayerCanvas;

#include <libgnomecanvasmm.h>
#include <libgnomecanvasmm/polygon.h>
#include "map_value.h"

///One tab in the Notebook that is placed in the main window (\ref MainWin).

///One digraph and all of its accessories like maps are assigned to one tab in the notebook.
///\ref NoteBookTab is responsible for the user defined display of the digraph: view can be
///set by visualized maps, therefore \ref NoteBookTab must provide an interface to set the
///view of digraph. This is \ref Mapwin window.
///
///\ref NoteBookTab is also
///responsible for modify the digraph if it is
///requested. Therefore it is responsible for translating user events to modifications to
///do on digraph, like node/arc addition/deletion, map modification, addition and so on.
///
///To be able to solve these tasks the help of \ref MainWin is also needed, for example to
///know which editor-tool is active at the moment. Therefore \ref MainWin knows \ref NoteBookTab.
///
///Some information in the other direction is needed as well: for example when new map creation is requested for this tab
///\ref NoteBookTab must ask \ref MainWin to pop-up a \ref NewMapWin. Communication in this direction is realized by signals
///therefore \ref NoteBookTab does not know \ref MainWin at all, but in this way it is not necessary.
class NoteBookTab : public Gtk::VBox
{
public:
  
  ///Constructor of \ref NoteBookTab

  ///It initiates the \re DigraphDisplayerCanvas, on which the digraph will be drawn
  ///Signals of \ref MapStorage will be bound to the appropriate callback functions here.
  NoteBookTab();
  
  ~NoteBookTab();
  
  ///Maps assigned to the digraph displayed in this \ref NoteBookTab of notebook.
  MapStorage * mapstorage;
    
  ///Title changement indicator.

  ///If digraph is loaded from disk or saved to disk or changed its name somehow
  ///this signal will be emit to let
  ///\ref MainWin know that it has to modify the title of the main window.
  ///It contains the new title.
  sigc::signal<void, std::string> signal_title;

  ///Returns \ref signal_title to be the caller able to connect it to a callback function.
  sigc::signal<void, std::string> signal_title_ch();

  ///Indicates that new map window should be popped up.

  ///\ref NoteBookTab can ask \ref MainWin to pop up a \ref NweMapWin ny emitting this signal.
  ///The signal contains whether an arcmap or a nodemap should be popped up. \ref NewMapWin
  ///is not popped up by \ref NoteBookTab, because not only \ref NoteBookTab needs \ref NewMapWin,
  ///but for example \ref MainWin and \ref AlgoWin s as well.
  sigc::signal<void, NoteBookTab *, bool> signal_newmap;

  ///Returns \ref signal_newmap to be the caller able to connect it to a callback function.
  sigc::signal<void, NoteBookTab *, bool> signal_newmap_needed();

  ///Loads the given file.

  ///The given file will be load in the \ref MapStorage and afeter that
  ///\ref DigraphDisplayerCanvas will be requested to display the digraph.
  ///\ref DigraphDisplayer will get datas from the recently set \ref MapStorage.
  void readFile(const std::string &);
    
  ///The digraph will be drawn on this \ref DigraphDisplayerCanvas
  DigraphDisplayerCanvas * gd_canvas;

  ///Indicates whether the \ref MapWin is opened or not. See \ref mapwin.
  bool mapwinexists;

  ///Indicates whether the \ref DesignWin is opened or not. See \ref designwin.
  bool designwinexists;

  ///Indicates whether the \ref EpsWin is opened or not. See \ref epswin.
  bool epswinexists;

  ///Address of the only \ref MapWin that the \ref NoteBookTab can open.

  ///Only one of this window can be opened at the same time (\ref mapwinexists), 
  ///because there is no need for more, one per tab is enough.
  ///There won1t be benefit of more than one, but it would be
  ///more complicated to synchronize them.
  MapWin * mapwin;
    
  ///Address of the only \ref DesignWin that the \ref NoteBookTab can open.

  ///Only one of this window can be opened at the same time (\ref designwinexists), 
  ///because there is no need for more, one per tab is enough.
  ///There won't be benefit of more than one, but it would be
  ///more complicated to synchronize them.
  DesignWin * designwin;

  ///Address of the only \ref EpsWin that the \ref NoteBookTab can open.

  ///Only one of this window can be opened at the same time (\ref epswinexists), 
  ///because there is no need for more, one per tab is enough.
  ///There won't be benefit of more than one.
  EpsWin * epswin;

public:
  ///Callback for 'FileNew' action.
  virtual void newFile();
  ///Callback for 'FileOpen' action.
  virtual void openFile();
  ///Callback for 'FileSave' action.
  virtual void saveFile();
  ///Callback for 'FileSaveAs' action.
  virtual void saveFileAs();
  ///Callback for 'Close' action.
  virtual void close();

  ///Handles changement of view of digraph.

  ///If the user changes the map to show by a property to a nother in \ref MapWin,
  ///\ref MapWin will call this function. This function will registrate in \ref MapStorage
  ///the new map to display by the changed property. After that \ref MapStorage will
  ///emits a signal that will be forwarded to \ref DigraphDisplayerCanvas to update the
  ///appropriate parts of digraph.
  ///\param itiesarc whether the changed property is arc property or node property
  ///\param prop id of property, see all_include.h
  ///\param mapname name of the recently selected map
  void propertyChange(bool itisarc, int prop, std::string mapname);

  ///Emits a signal that request \ref MainWin to pop up \ref NewMapWin

  ///This function is called by \ref MapWin.
  ///\param itisarc whether the new map should be an arcmap or a nodemap.
  void popupNewMapWin(bool itisarc);

  ///Returns the actually selected arcmap to visualize by the given property.

  ///\ref MapWin calls this function, beacuse it does not know \ref MapStorage.
  ///\param prop property to inquire.
  std::string getActiveArcMap(int prop);

  ///Returns the actually selected nodemap to visualize by the given property.

  ///\ref MapWin calls this function, beacuse it does not know \ref MapStorage.
  ///\param prop property to inquire.
  std::string getActiveNodeMap(int prop);

  ///Registers recently created arcmap in \ref MapWin.

  ///After creation of new map \ref MapStorage emits a signal.
  ///This signal is bound to this callback function, that will call
  ///a function with the same name and same parameterin \ref MapWin.
  ///This call-forwarder function is needed, because \ref Mapstorage does not know \ref MapWin
  ///\param mapname name of new map
  void registerNewArcMap(std::string mapname, MapValue::Type type);

  ///Registers recently created nodemap in \ref MapWin.

  ///After creation of new map \ref MapStorage emits a signal.
  ///This signal is bound to this callback function, that will call
  ///a function with the same name and same parameter in \ref MapWin.
  ///This call-forwarder function is needed, because \ref Mapstorage does not know \ref MapWin
  ///\param mapname name of new map
  void registerNewNodeMap(std::string mapname, MapValue::Type type);

  ///Pops up and registrates the \ref MapWin of \ref NoteBookTab.
  
  ///See also
  ///\ref mapwin.
  void createMapWin(std::string);

  ///Pops up and registrates the \ref DesignWin of \ref NoteBookTab.
  
  ///See also
  ///\ref mapwin.
  void createDesignWin(std::string);

  ///Pops up a window, that can dump digraph to EPS

  ///Different parameters can be set here.
  void createExportToEPSWin(std::string);

  ///Closes and deregistrates the \ref MapWin of \ref NoteBookTab.
  
  ///See also
  ///\ref mapwin.
  void closeMapWin();

  ///Closes and deregistrates the \ref DesignWin of \ref NoteBookTab.
  
  ///See also
  ///\ref designwin.
  bool closeDesignWin(GdkEventAny *);

  ///Closes and deregistrates the \ref EpsWin of \ref NoteBookTab.
  
  ///See also
  ///\ref epswin.
  void closeEpsWin();

  ///Sets node representation settings
  void setView(bool, bool, double, double);

  ///Gets node representation settings
  void getView(bool &, bool &, double&, double&);

  ///Let the digraph redesign, based on gravity and arc elasticity.
  void reDesignDigraph();

  ///Lets Mapstorage export the digraph to EPS
  void exportDigraphToEPS(std::vector<bool>, std::string, std::string);

  ///\ref MapWin calls this function when it updates the maplist in comboboxes.
  void active_maps_needed();

private:
  ///Called when title of tab has changed
  void title_changed(std::string);

  ///Signal connection from \ref MapStorage to \ref MapWin

  ///If \ref MapWin is closed this connection has to be disconnected,
  ///therefore we have to store it.
  sigc::connection mapst2mapwin;

  ///Signal connection from \ref MapStorage to \ref DesignWin

  ///If \ref DesignWin is closed this connection has to be disconnected,
  ///therefore we have to store it.
  sigc::connection mapst2designwin;
};

#endif //NBTAB_H
