/* -*- C++ -*-
 *
 * This file is a part of LEMON, a generic C++ optimization library
 *
 * Copyright (C) 2003-2006
 * Egervary Jeno Kombinatorikus Optimalizalasi Kutatocsoport
 * (Egervary Research Group on Combinatorial Optimization, EGRES).
 *
 * Permission to use, modify and distribute this software is granted
 * provided that this copyright notice appears in all copies. For
 * precise terms see the accompanying LICENSE file.
 *
 * This software is provided "AS IS" with no warranty of any kind,
 * express or implied, and with no claim as to its suitability for any
 * purpose.
 *
 */

#ifndef NEWMAPWIN_H
#define NEWMAPWIN_H

#include <all_include.h>
#include <libgnomecanvasmm.h>
#include <libgnomecanvasmm/polygon.h>
#include <stack>

class NoteBookTab;

///Digraphical interface for node/arc map creation.

///This class is responsible for creating a window,
///on which the parameters of a new map can be set.
class NewMapWin : public Gtk::Dialog
{
  ///The \ref NoteBookTab in which the new map has to be placed.
  NoteBookTab & mytab;

  MapType map_type;
  Gtk::Label lblType;
  Gtk::ComboBoxText cbType;

  Gtk::Label lblErrorMsg;
  void setErrorMsg(const Glib::ustring& msg);

  std::vector<double>* evaluate_expr(const std::string polishform, bool itisarc);

public:

  ///Struct to be able to evaluate expressions.

  ///Initial values of map elements can be given
  ///by numbers or by expressions. From expressions
  ///we build first a tree according to the priorities
  ///of operations in the expression. This is the data
  ///structure
  ///that can store one tree element.
  struct tree_node
  {
    ///Character stored in this tree element
    char ch;
    ///Left child of tree element.
    tree_node * left_child;
    ///Right child of tree element.
    tree_node * right_child;
  };
  
  ///Constructor of NewMapWin.

  ///It creates the widgets shown in
  ///NewMapWin.
  NewMapWin(const std::string& title, NoteBookTab &, bool itisarc=true, bool arcnode=true, MapType type = ALL);

  ///Callback function for OK button. It creates the map.
  
  ///This function determines whether the input is correct:
  ///the name of new map must not be already used, the expression
  ///that gives tha initial values of map elements has to be valid.
  ///If input is correct it creates and registrates the new map
  ///to the correct place. (\ref mytab)
  virtual void on_response(int response_id);

  ///Close window if Esc key pressed.
  virtual bool closeIfEscapeIsPressed(GdkEventKey*);

  ///Function that creates a tree from an appropriately manipulated string.

  ///Tree is builded according to priorities of operations in expression given by string.
  ///Priorities are indicated in a vector that contains weights for each operation.
  ///\param to_tree string to build tree from
  ///\param weights weights (priorities)
  ///\param offset this function call is recursive. This parameter tells us,
  ///with which part of the string do we have to deal with.
  tree_node * weightedString2Tree(std::string to_tree, std::vector<unsigned int> & weights, int offset);

  ///Function that creates a string from a tree by postorder reading.

  ///This is the last step of creating polishform
  ///from a given expression string.
  ///\param root the root of the tree to read through
  std::string postOrder(tree_node * root);

  ///From the given expression it creates expression given in polish form.

  ///First it substitutes variables and numbers in the given expression.
  ///The substitutions will be one character long local variables.
  ///The substituted-substitution pair is registrated in \ref ch2var.
  ///After that it gives weights fo each character in substituted expression.
  ///Weights are calculated according to the priority of operations in expression.
  ///Then it creates tree (\ref tree_node) from the weighted string. (\ref weightedString2Tree)
  ///\param to_polish the string to turn into polish_form
  ///\param itisarc do we have to create an arcmap or a nodemap.
  ///It is important, because variables are maps and if expression
  ///citates a map that does not exists the expression is not valid.
  ///But to determine, whether the map exists we have to know where
  ///to search for it. And of course for a new arcmap only arcmaps can be serve with values. 
  std::string string2Polishform(std::string to_polish, bool itisarc);

  ///Returns whether a string can be used as value in an expression.

  ///The given string has to be either a mapname or a number. If it is a mapname
  ///we have to know whether it is an arcmap or a nodemap.
  ///\param variable the string about the function has to determine whether it is usable in expressions
  ///\param itisarc should the mapname be between arcmaps, or nodemaps
  bool validVariable(std::string variable, bool itisarc);

  ///Deletes the whole tree created for translating string to polishform.
 
  ///\param root
  ///root of the tree
  void deleteTree(tree_node * root);

  ///Dictionary of substitutions in expression.

  ///Variables and numbers are substituted with one character long variables in expressions.
  ///This is the dictionary.
  std::map<char, std::string> ch2var;

  ///Entry which gives us the name of new map.
  Gtk::Entry name;

  ///Entry which gives us the initial values of elements of new map.

  ///Initial value can be a number or an expression after that the
  ///initial value for each map element can be calculated.
  Gtk::Entry default_value;

  ///GTK Designing object.
  Gtk::Table * table;

  ///Information holder in window.
  Gtk::Label * label;

  ///If selected, nodemap will be created.
  Gtk::RadioButton node;

  ///If selected, arcmap will be created.
  Gtk::RadioButton arc;
};

#endif //NEWMAPWIN_H
