#ifndef LIST_GRAPH_HH
#define LIST_GRAPH_HH

#include <iostream>

namespace marci {

  template <typename It>
  int count(It it) { 
    int i=0;
    for( ; it.valid(); ++it) { ++i; } 
    return i;
  }

  class ListGraph {

    class node_item;
    class edge_item;

  public:

    class NodeIt;
    class EachNodeIt;
    class EdgeIt;
    class EachEdgeIt;
    class OutEdgeIt;
    class InEdgeIt;
    class SymEdgeIt;
    template <typename ValueType> class NodeMap;
    template <typename ValueType> class EdgeMap;
    
  private:
    
    template <typename ValueType> friend class NodeMap;
    template <typename ValueType> friend class EdgeMap;

    template <typename ValueType>
    class NodeMap {
      const ListGraph& G; 
      std::vector<ValueType> container;
    public:
      NodeMap(const ListGraph& _G) : G(_G), container(_G.node_id) { }
      NodeMap(const ListGraph& _G, ValueType a) : 
	G(_G), container(_G.node_id, a) { }
      void set(NodeIt nit, ValueType a) { container[G.id(nit)]=a; }
      ValueType get(NodeIt nit) const { return container[G.id(nit)]; }
    };

    template <typename ValueType>
    class EdgeMap {
      const ListGraph& G; 
      std::vector<ValueType> container;
    public:
      EdgeMap(const ListGraph& _G) : G(_G), container(_G.edge_id) { }
      EdgeMap(const ListGraph& _G, ValueType a) : 
	G(_G), container(_G.edge_id, a) { }
      void set(EdgeIt eit, ValueType a) { container[G.id(eit)]=a; }
      ValueType get(EdgeIt eit) const { return container[G.id(eit)]; }
    };

    int node_id;
    int edge_id;
    int _node_num;
    int _edge_num;

    node_item* _first_node;
    node_item* _last_node;

    class node_item {
      friend class ListGraph;
      friend class NodeIt;
      friend class EachNodeIt;
      friend class EdgeIt;
      friend class EachEdgeIt;
      friend class OutEdgeIt;
      friend class InEdgeIt;
      friend class SymEdgeIt;
      friend std::ostream& operator<<(std::ostream& os, const NodeIt& i);
      friend std::ostream& operator<<(std::ostream& os, const EdgeIt& i);
      //ListGraph* G;
      int id;
      edge_item* _first_out_edge;
      edge_item* _last_out_edge;
      edge_item* _first_in_edge;
      edge_item* _last_in_edge;
      node_item* _next_node;
      node_item* _prev_node;
    public:
      node_item() { }
    };

    class edge_item {
      friend class ListGraph;
      friend class NodeIt;
      friend class EachNodeIt;
      friend class EdgeIt;
      friend class EachEdgeIt;
      friend class OutEdgeIt;
      friend class InEdgeIt;
      friend class SymEdgeIt;
      friend std::ostream& operator<<(std::ostream& os, const EdgeIt& i);
      //ListGraph* G;
      int id;
      node_item* _tail;
      node_item* _head;
      edge_item* _next_out;
      edge_item* _prev_out;
      edge_item* _next_in;
      edge_item* _prev_in;
    public:
      edge_item() { }
    };

    node_item* _add_node() { 
      node_item* p=new node_item;
      p->id=node_id++;
      p->_first_out_edge=0;
      p->_last_out_edge=0;
      p->_first_in_edge=0;
      p->_last_in_edge=0;
      p->_prev_node=_last_node;
      p->_next_node=0;
      if (_last_node) _last_node->_next_node=p;
      _last_node=p;
      if (!_first_node) _first_node=p;

      ++_node_num;
      return p;
    }

    edge_item* _add_edge(node_item* _tail, node_item* _head) {
      edge_item* e=new edge_item;
      e->id=edge_id++;
      e->_tail=_tail;
      e->_head=_head;
      
      e->_prev_out=_tail->_last_out_edge;
      if (_tail->_last_out_edge) (_tail->_last_out_edge)->_next_out=e;
      _tail->_last_out_edge=e;
      if (!_tail->_first_out_edge) _tail->_first_out_edge=e; 
      e->_next_out=0;
 
      e->_prev_in=_head->_last_in_edge;
      if (_head->_last_in_edge) (_head->_last_in_edge)->_next_in=e;
      _head->_last_in_edge=e;
      if (!_head->_first_in_edge) { _head->_first_in_edge=e; } 
      e->_next_in=0;

      ++_edge_num;
      return e;
    }

    //deletes a node which has no out edge and no in edge
    void _delete_node(node_item* v) {
      if (v->_next_node) (v->_next_node)->_prev_node=v->_prev_node; else 
	_last_node=v->_prev_node;
      if (v->_prev_node) (v->_prev_node)->_next_node=v->_next_node; else 
	_first_node=v->_next_node;

      delete v;
      --_node_num;
    }

    void _delete_edge(edge_item* e) {
      if (e->_next_out) (e->_next_out)->_prev_out=e->_prev_out; else 
	(e->_tail)->_last_out_edge=e->_prev_out;
      if (e->_prev_out) (e->_prev_out)->_next_out=e->_next_out; else 
	(e->_tail)->_first_out_edge=e->_next_out;
      if (e->_next_in) (e->_next_in)->_prev_in=e->_prev_in; else 
	(e->_head)->_last_in_edge=e->_prev_in;
      if (e->_prev_in) (e->_prev_in)->_next_in=e->_next_in; else 
	(e->_head)->_first_in_edge=e->_next_in;

      delete e;
      --_edge_num;
    }

    void _set_tail(edge_item* e, node_item* _tail) {
      if (e->_next_out) (e->_next_out)->_prev_out=e->_prev_out; else 
	(e->_tail)->_last_out_edge=e->_prev_out;
      if (e->_prev_out) (e->_prev_out)->_next_out=e->_next_out; else 
	(e->_tail)->_first_out_edge=e->_next_out;
      
      e->_tail=_tail;
      
      e->_prev_out=_tail->_last_out_edge;
      if (_tail->_last_out_edge) (_tail->_last_out_edge)->_next_out=e;
      _tail->_last_out_edge=e;
      if (!_tail->_first_out_edge) _tail->_first_out_edge=e; 
      e->_next_out=0;
    }

    void _set_head(edge_item* e, node_item* _head) {
      if (e->_next_in) (e->_next_in)->_prev_in=e->_prev_in; else 
	(e->_head)->_last_in_edge=e->_prev_in;
      if (e->_prev_in) (e->_prev_in)->_next_in=e->_next_in; else 
	(e->_head)->_first_in_edge=e->_next_in;
      
      e->_head=_head;
      
      e->_prev_in=_head->_last_in_edge;
      if (_head->_last_in_edge) (_head->_last_in_edge)->_next_in=e;
      _head->_last_in_edge=e;
      if (!_head->_first_in_edge) { _head->_first_in_edge=e; } 
      e->_next_in=0;
    }

  public:

    /* default constructor */

    ListGraph() : node_id(0), edge_id(0), _node_num(0), _edge_num(0), _first_node(0), _last_node(0) { }
    
    ~ListGraph() { 
      while (first<EachNodeIt>().valid()) deleteNode(first<EachNodeIt>());
    }

    int nodeNum() const { return _node_num; }
    int edgeNum() const { return _edge_num; }

    /* functions to construct iterators from the graph, or from each other */

    //EachNodeIt firstNode() const { return EachNodeIt(*this); }
    //EachEdgeIt firstEdge() const { return EachEdgeIt(*this); }
    
    //OutEdgeIt firstOutEdge(const NodeIt v) const { return OutEdgeIt(v); }
    //InEdgeIt firstInEdge(const NodeIt v) const { return InEdgeIt(v); }
    //SymEdgeIt firstSymEdge(const NodeIt v) const { return SymEdgeIt(v); }
    NodeIt tail(EdgeIt e) const { return e.tailNode(); }
    NodeIt head(EdgeIt e) const { return e.headNode(); }

    NodeIt aNode(const OutEdgeIt& e) const { return e.aNode(); }
    NodeIt aNode(const InEdgeIt& e) const { return e.aNode(); }
    NodeIt aNode(const SymEdgeIt& e) const { return e.aNode(); }

    NodeIt bNode(const OutEdgeIt& e) const { return e.bNode(); }
    NodeIt bNode(const InEdgeIt& e) const { return e.bNode(); }
    NodeIt bNode(const SymEdgeIt& e) const { return e.bNode(); }

    //NodeIt invalid_node() { return NodeIt(); }
    //EdgeIt invalid_edge() { return EdgeIt(); }
    //OutEdgeIt invalid_out_edge() { return OutEdgeIt(); }
    //InEdgeIt invalid_in_edge() { return InEdgeIt(); }
    //SymEdgeIt invalid_sym_edge() { return SymEdgeIt(); }

    /* same methods in other style */
    /* for experimental purpose */

    void getFirst(EachNodeIt& v) const { v=EachNodeIt(*this); }
    void getFirst(EachEdgeIt& e) const { e=EachEdgeIt(*this); }
    void getFirst(OutEdgeIt& e, NodeIt v) const { e=OutEdgeIt(v); }
    void getFirst(InEdgeIt& e, NodeIt v) const { e=InEdgeIt(v); }
    void getFirst(SymEdgeIt& e, NodeIt v) const { e=SymEdgeIt(v); }
    //void getTail(NodeIt& n, const EdgeIt& e) const { n=tail(e); }
    //void getHead(NodeIt& n, const EdgeIt& e) const { n=head(e); }

    //void getANode(NodeIt& n, const OutEdgeIt& e) const { n=e.aNode(); }
    //void getANode(NodeIt& n, const InEdgeIt& e) const { n=e.aNode(); }
    //void getANode(NodeIt& n, const SymEdgeIt& e) const { n=e.aNode(); }
    //void getBNode(NodeIt& n, const OutEdgeIt& e) const { n=e.bNode(); }
    //void getBNode(NodeIt& n, const InEdgeIt& e) const { n=e.bNode(); }
    //void getBNode(NodeIt& n, const SymEdgeIt& e) const { n=e.bNode(); }
    //void get_invalid(NodeIt& n) { n=NodeIt(); }
    //void get_invalid(EdgeIt& e) { e=EdgeIt(); }
    //void get_invalid(OutEdgeIt& e) { e=OutEdgeIt(); }
    //void get_invalid(InEdgeIt& e) { e=InEdgeIt(); }
    //void get_invalid(SymEdgeIt& e) { e=SymEdgeIt(); }

    template< typename It >
    It first() const { 
      It e;
      getFirst(e);
      return e; 
    }

    template< typename It >
    It first(NodeIt v) const { 
      It e;
      getFirst(e, v);
      return e; 
    }

    /* for getting id's of graph objects */
    /* these are important for the implementation of property vectors */

    int id(NodeIt v) const { return v.node->id; }
    int id(EdgeIt e) const { return e.edge->id; }

    /* adding nodes and edges */

    NodeIt addNode() { return NodeIt(_add_node()); }
    EdgeIt addEdge(NodeIt u, NodeIt v) {
      return EdgeIt(_add_edge(u.node, v.node)); 
    }

    void deleteNode(NodeIt i) { 
      while (first<OutEdgeIt>(i).valid()) deleteEdge(first<OutEdgeIt>(i));
      while (first<InEdgeIt>(i).valid()) deleteEdge(first<InEdgeIt>(i));
      _delete_node(i.node); 
    }
  
    void deleteEdge(EdgeIt e) { _delete_edge(e.edge); }

    void setTail(EdgeIt e, NodeIt tail) {
      _set_tail(e.edge, tail.node); 
    }

    void setHead(EdgeIt e, NodeIt head) {
      _set_head(e.edge, head.node); 
    }

    /* stream operations, for testing purpose */

    friend std::ostream& operator<<(std::ostream& os, const NodeIt& i) { 
      os << i.node->id; return os; 
    }
    friend std::ostream& operator<<(std::ostream& os, const EdgeIt& i) { 
      os << "(" << i.edge->_tail->id << "--" << i.edge->id << "->" << i.edge->_head->id << ")"; 
      return os; 
    }

    class NodeIt {
      friend class ListGraph;

      friend class EdgeIt;
      friend class OutEdgeIt;
      friend class InEdgeIt;
      friend class SymEdgeIt;
    protected:
      node_item* node;
      friend int ListGraph::id(NodeIt v) const; 
    public:
      NodeIt() : node(0) { }
      NodeIt(node_item* _node) : node(_node) { }
      bool valid() const { return (node!=0); }
      //void makeInvalid() { node=0; }
      friend bool operator==(const NodeIt& u, const NodeIt& v) { 
	return v.node==u.node; 
      } 
      friend bool operator!=(const NodeIt& u, const NodeIt& v) { 
	return v.node!=u.node; 
      } 
      friend std::ostream& operator<<(std::ostream& os, const NodeIt& i);
    };
    
    class EachNodeIt : public NodeIt {
      friend class ListGraph;
    protected:
      EachNodeIt(const ListGraph& G) : NodeIt(G._first_node) { }
    public:
      EachNodeIt() : NodeIt() { }
      EachNodeIt(node_item* v) : NodeIt(v) { }
      EachNodeIt& operator++() { node=node->_next_node; return *this; }
    };

    class EdgeIt {
      friend class ListGraph;
      
      friend class NodeIt;
      friend class EachNodeIt;
    protected:
      edge_item* edge;
      friend int ListGraph::id(EdgeIt e) const;
    public:
      EdgeIt() : edge(0) { }
      //EdgeIt() { }
      EdgeIt(edge_item* _edge) : edge(_edge) { }
      bool valid() const { return (edge!=0); }
      //void makeInvalid() { edge=0; }
      friend bool operator==(const EdgeIt& u, const EdgeIt& v) { 
	return v.edge==u.edge; 
      } 
      friend bool operator!=(const EdgeIt& u, const EdgeIt& v) { 
	return v.edge!=u.edge; 
      } 
    protected:
      NodeIt tailNode() const { return NodeIt(edge->_tail); }
      NodeIt headNode() const { return NodeIt(edge->_head); }
    public:
      friend std::ostream& operator<<(std::ostream& os, const EdgeIt& i);
    };
    
    class EachEdgeIt : public EdgeIt {
      friend class ListGraph;
    protected:
      EachEdgeIt(const ListGraph& G) {
	node_item* v=G._first_node;
	if (v) edge=v->_first_out_edge; else edge=0;
	while (v && !edge) { v=v->_next_node; if (v) edge=v->_first_out_edge; }
      }
    public:
      EachEdgeIt() : EdgeIt() { }
      EachEdgeIt(edge_item* _e) : EdgeIt(_e) { }
      EachEdgeIt& operator++() { 
	node_item* v=edge->_tail;
	edge=edge->_next_out; 
	while (v && !edge) { v=v->_next_node; if (v) edge=v->_first_out_edge; }
	return *this;
      }
    };
    
    class OutEdgeIt : public EdgeIt {
      friend class ListGraph;
      node_item* v;
    protected:
      OutEdgeIt(const NodeIt& _v) : v(_v.node) { edge=v->_first_out_edge; }
    public:
      OutEdgeIt() : EdgeIt(), v(0) { }
      OutEdgeIt(const ListGraph& G, NodeIt _v) : v(_v.node) { edge=v->_first_out_edge; }
      OutEdgeIt& operator++() { edge=edge->_next_out; return *this; }
    protected:
      NodeIt aNode() const { return NodeIt(v); }
      NodeIt bNode() const { 
	return (edge->_tail==v) ? NodeIt(edge->_head) : NodeIt(edge->_tail); }
    };
    
    class InEdgeIt : public EdgeIt {
      friend class ListGraph;
      node_item* v;
    protected:
      InEdgeIt(const NodeIt& _v) : v(_v.node) { edge=v->_first_in_edge; }
    public:
      InEdgeIt() : EdgeIt(), v(0) { }
      InEdgeIt(const ListGraph& G, NodeIt _v) : v(_v.node) { edge=v->_first_in_edge; }
      InEdgeIt& operator++() { edge=edge->_next_in; return *this; }
    protected:
      NodeIt aNode() const { return NodeIt(v); }
      NodeIt bNode() const { 
	return (edge->_tail==v) ? NodeIt(edge->_head) : NodeIt(edge->_tail); }
    };

    class SymEdgeIt : public EdgeIt {
      friend class ListGraph;
      bool out_or_in; //1 iff out, 0 iff in
      node_item* v;
    protected:
      SymEdgeIt(const NodeIt& _v) : v(_v.node) { 
	out_or_in=1;
	edge=v->_first_out_edge; 
	if (!edge) { edge=v->_first_in_edge; out_or_in=0; }
      }
    public:
      SymEdgeIt() : EdgeIt(), v(0) { }
      SymEdgeIt(const ListGraph& G, NodeIt _v) : v(_v.node) { 
	out_or_in=1;
	edge=v->_first_out_edge; 
	if (!edge) { edge=v->_first_in_edge; out_or_in=0; }
      }
      SymEdgeIt& operator++() { 
	if (out_or_in) { 
	  edge=edge->_next_out; 
	  if (!edge) { out_or_in=0; edge=v->_first_in_edge; }
	} else {
	  edge=edge->_next_in; 
	}
	return *this;
      }
    protected:
      NodeIt aNode() const { return NodeIt(v); }
      NodeIt bNode() const { 
	return (edge->_tail==v) ? NodeIt(edge->_head) : NodeIt(edge->_tail); }
    };

  };

  /*
  template< typename T >
  T ListGraph::first() const { 
    std::cerr << "Invalid use of template<typemane T> T ListGraph::first<T>();" << std::endl; 
    return T(); 
  }

  template<>
  ListGraph::EachNodeIt ListGraph::first<ListGraph::EachNodeIt>() const { 
    return firstNode(); 
  }

  template<>
  ListGraph::EachEdgeIt ListGraph::first<ListGraph::EachEdgeIt>() const { 
    return firstEdge(); 
  }

  template< typename T >
  T ListGraph::first(ListGraph::NodeIt v) const {
    std::cerr << "Invalid use of template<typemane T> T ListGraph::first<T>(ListGRaph::NodeIt);" << std::endl; 
    return T(); 
  } 

  template<>
  ListGraph::OutEdgeIt ListGraph::first<ListGraph::OutEdgeIt>(const ListGraph::NodeIt v) const { 
    return firstOutEdge(v); 
  }

  template<>
  ListGraph::InEdgeIt ListGraph::first<ListGraph::InEdgeIt>(const ListGraph::NodeIt v) const { 
    return firstInEdge(v); 
  }

  template<>
  ListGraph::SymEdgeIt ListGraph::first<ListGraph::SymEdgeIt>(const ListGraph::NodeIt v) const { 
    return firstSymEdge(v); 
  }
  */


} //namespace marci

#endif //LIST_GRAPH_HH
