/* -*- C++ -*-
 * src/test/preflow_test.cc - Part of LEMON, a generic C++ optimization library
 *
 * Copyright (C) 2005 Egervary Jeno Kombinatorikus Optimalizalasi Kutatocsoport
 * (Egervary Combinatorial Optimization Research Group, EGRES).
 *
 * Permission to use, modify and distribute this software is granted
 * provided that this copyright notice appears in all copies. For
 * precise terms see the accompanying LICENSE file.
 *
 * This software is provided "AS IS" with no warranty of any kind,
 * express or implied, and with no claim as to its suitability for any
 * purpose.
 *
 */

#include <fstream>
#include <string>

#include "test_tools.h"
#include <lemon/smart_graph.h>
#include <lemon/dimacs.h>
#include <lemon/preflow.h>
#include <lemon/concept/graph.h>
#include <lemon/concept/maps.h>

using namespace lemon;

void check_Preflow() 
{
  typedef int VType;
  typedef concept::StaticGraph Graph;

  typedef Graph::Node Node;
  typedef Graph::Edge Edge;
  typedef concept::ReadMap<Edge,VType> CapMap;
  typedef concept::ReadWriteMap<Edge,VType> FlowMap;
  typedef concept::ReadWriteMap<Node,bool> CutMap;
 
  typedef Preflow<Graph, int, CapMap, FlowMap> PType;

  Graph g;
  Node n;
  CapMap cap;
  FlowMap flow;
  CutMap cut;

  PType preflow_test(g,n,n,cap,flow);

  preflow_test.run();
  preflow_test.flowValue();
  preflow_test.source(n);
  preflow_test.flowMap(flow);

  preflow_test.phase1(PType::NO_FLOW);
  preflow_test.minCut(cut);

  preflow_test.phase2();
  preflow_test.target(n);
  preflow_test.capacityMap(cap);
  preflow_test.minMinCut(cut);
  preflow_test.maxMinCut(cut);
}

int cut_value ( SmartGraph& g, SmartGraph::NodeMap<bool>& cut, 
		SmartGraph::EdgeMap<int>& cap) {
  
  int c=0;
  for(SmartGraph::EdgeIt e(g); e!=INVALID; ++e) {
    if (cut[g.source(e)] && !cut[g.target(e)]) c+=cap[e];
  }
  return c;
}

int main() {

  typedef SmartGraph Graph;
  
  typedef Graph::Node Node;
  typedef Graph::NodeIt NodeIt;
  typedef Graph::EdgeIt EdgeIt;
  typedef Graph::EdgeMap<int> CapMap;
  typedef Graph::EdgeMap<int> FlowMap;
  typedef Graph::NodeMap<bool> CutMap;

  typedef Preflow<Graph, int> PType;

  std::string f_name;
  if( getenv("srcdir") )
    f_name = std::string(getenv("srcdir"));
  else f_name = ".";
  f_name += "/preflow_graph.dim";
  
  std::ifstream file(f_name.c_str());
  
  check(file, "Input file '" << f_name << "' not found.");
  
  Graph g;
  Node s, t;
  CapMap cap(g);
  readDimacs(file, g, cap, s, t);

  FlowMap flow(g,0);

 

  PType preflow_test(g, s, t, cap, flow);
  preflow_test.run(PType::ZERO_FLOW);
    
  CutMap min_cut(g,false);
  preflow_test.minCut(min_cut); 
  int min_cut_value=cut_value(g,min_cut,cap);
   
  CutMap min_min_cut(g,false);
  preflow_test.minMinCut(min_min_cut); 
  int min_min_cut_value=cut_value(g,min_min_cut,cap);
   
  CutMap max_min_cut(g,false);
  preflow_test.maxMinCut(max_min_cut); 
  int max_min_cut_value=cut_value(g,max_min_cut,cap);

  check(preflow_test.flowValue() == min_cut_value &&
	min_cut_value == min_min_cut_value &&
	min_min_cut_value == max_min_cut_value,
	"The max flow value is not equal to the three min cut values.");

  int flow_value=preflow_test.flowValue();



  for(EdgeIt e(g); e!=INVALID; ++e) cap[e]=2*cap[e]; 
  preflow_test.capacityMap(cap);  

  preflow_test.phase1(PType::PRE_FLOW);

  CutMap min_cut1(g,false);
  preflow_test.minCut(min_cut1); 
  min_cut_value=cut_value(g,min_cut1,cap);
   
  check(preflow_test.flowValue() == min_cut_value &&
	min_cut_value == 2*flow_value,
	"The max flow value or the min cut value is wrong.");

  preflow_test.phase2();

  CutMap min_cut2(g,false);
  preflow_test.minCut(min_cut2); 
  min_cut_value=cut_value(g,min_cut2,cap);
   
  CutMap min_min_cut2(g,false);
  preflow_test.minMinCut(min_min_cut2); 
  min_min_cut_value=cut_value(g,min_min_cut2,cap);
 
  preflow_test.maxMinCut(max_min_cut); 
  max_min_cut_value=cut_value(g,max_min_cut,cap);

  check(preflow_test.flowValue() == min_cut_value &&
	min_cut_value == min_min_cut_value &&
	min_min_cut_value == max_min_cut_value &&
	min_cut_value == 2*flow_value,
	"The max flow value or the three min cut values were not doubled");



  EdgeIt e(g);
  for( int i=1; i==10; ++i ) {
    flow.set(e,0);
    ++e;
  }

  preflow_test.flowMap(flow); 

  NodeIt tmp1(g,s);
  ++tmp1;
  if ( tmp1 != INVALID ) s=tmp1;

  NodeIt tmp2(g,t);
  ++tmp2;
  if ( tmp2 != INVALID ) t=tmp2;

  preflow_test.source(s);
  preflow_test.target(t); 
  
  preflow_test.run();

  CutMap min_cut3(g,false);
  preflow_test.minCut(min_cut3); 
  min_cut_value=cut_value(g,min_cut3,cap);
   
  CutMap min_min_cut3(g,false);
  preflow_test.minMinCut(min_min_cut3); 
  min_min_cut_value=cut_value(g,min_min_cut3,cap);
   
  preflow_test.maxMinCut(max_min_cut); 
  max_min_cut_value=cut_value(g,max_min_cut,cap);

  check(preflow_test.flowValue() == min_cut_value &&
	min_cut_value == min_min_cut_value &&
	min_min_cut_value == max_min_cut_value,
	"The max flow value or the three min cut values are incorrect.");
}
