// -*- C++ -*-
/*
preflow_push_hl.hh
by jacint. 
Runs the highest label variant of the preflow push algorithm with 
running time O(n^2\sqrt(m)). 

Member functions:

void run() : runs the algorithm

 The following functions should be used after run() was already run.

T maxflow() : returns the value of a maximum flow

T flowonEdge(Edge_iterator e) : for a fixed maximum flow x it returns x(e) 

EdgeMap<graph_type, T> allflow() : returns the fixed maximum flow x

NodeMap<graph_type, bool> mincut() : returns a 
     characteristic vector of a minimum cut. (An empty level 
     in the algorithm gives a minimum cut.)
*/

#ifndef PREFLOW_PUSH_HL_H
#define PREFLOW_PUSH_HL_H

#include <algorithm>
#include <vector>
#include <stack>

#include <reverse_bfs.hh>

namespace marci {

  template <typename Graph, typename T, typename FlowMap, typename CapacityMap>
  class preflow_push_hl {
    
    typedef typename Graph::NodeIt NodeIt;
    typedef typename Graph::EdgeIt EdgeIt;
    typedef typename Graph::EachNodeIt EachNodeIt;
    typedef typename Graph::OutEdgeIt OutEdgeIt;
    typedef typename Graph::InEdgeIt InEdgeIt;
    typedef typename Graph::EachEdgeIt EachEdgeIt;
    

    Graph& G;
    NodeIt s;
    NodeIt t;
    Graph::EdgeMap<T> flow;
    Graph::EdgeMap<T> capacity; 
    T value;
    Graph::NodeMap<bool> mincutvector;

   
  public:

    preflow_push_hl(Graph& _G, NodeIt _s, NodeIt _t, 
		    Graph::EdgeMap<T>& _capacity) :
      G(_G), s(_s), t(_t), flow(_G, 0), capacity(_capacity), mincutvector(_G, true) { }




    /*
      The run() function runs the highest label preflow-push, 
      running time: O(n^2\sqrt(m))
    */
    void run() {
 
      Graph::NodeMap<int> level(G);         //level of Node
      Graph::NodeMap<T> excess(G);          //excess of Node
            
      int n=G.nodeNum();                        //number of Nodes 
      int b=n; 
      /*b is a bound on the highest level of an active Node. In the beginning it is at most n-2.*/

      std::vector<std::stack<NodeIt> > stack(2*n-1);    //Stack of the active Nodes in level i.




      /*Reverse_bfs from t, to find the starting level.*/

      reverse_bfs<list_graph> bfs(G, t);
      bfs.run();
      for(EachNodeIt v=G.template first<EachNodeIt>(); v.valid(); ++v) {
	level.set(v, bfs.dist(v)); 
	//std::cout << "the level of " << v << " is " << bfs.dist(v);
      }

      /*The level of s is fixed to n*/ 
      level.set(s,n);





      /* Starting flow. It is everywhere 0 at the moment. */
     
      for(OutEdgeIt i=G.template first<OutEdgeIt>(s); i.valid(); ++i) 
	{
	  NodeIt w=G.head(i);
	  flow.set(i, capacity.get(i)); 
	  stack[bfs.dist(w)].push(w); 
	  excess.set(w, capacity.get(i));
	}


      /* 
	 End of preprocessing 
      */



      /*
	Push/relabel on the highest level active Nodes.
      */
	
      /*While there exists active Node.*/
      while (b) { 

	/*We decrease the bound if there is no active Node of level b.*/
	if (stack[b].empty()) {
	  --b;
	} else {

	  NodeIt w=stack[b].top();    //w is the highest label active Node.
	  stack[b].pop();                    //We delete w from the stack.
	
	  int newlevel=2*n-2;                   //In newlevel we maintain the next level of w.
	
	  for(OutEdgeIt e=G.template first<OutEdgeIt>(w); e.valid(); ++e) {
	    NodeIt v=G.head(e);
	    /*e is the Edge wv.*/

	    if (flow.get(e)<capacity.get(e)) {              
	      /*e is an Edge of the residual graph */

	      if(level.get(w)==level.get(v)+1) {      
		/*Push is allowed now*/

		if (capacity.get(e)-flow.get(e) > excess.get(w)) {       
		  /*A nonsaturating push.*/
		  
		  if (excess.get(v)==0 && v != s) stack[level.get(v)].push(v); 
		  /*v becomes active.*/
		  
		  flow.set(e, flow.get(e)+excess.get(w));
		  excess.set(v, excess.get(v)+excess.get(w));
		  excess.set(w,0);
		  //std::cout << w << " " << v <<" elore elen nonsat pump "  << std::endl;
		  break; 
		} else { 
		  /*A saturating push.*/

		  if (excess.get(v)==0 && v != s) stack[level.get(v)].push(v); 
		  /*v becomes active.*/

		  excess.set(v, excess.get(v)+capacity.get(e)-flow.get(e));
		  excess.set(w, excess.get(w)-capacity.get(e)+flow.get(e));
		  flow.set(e, capacity.get(e));
		  //std::cout << w<<" " <<v<<" elore elen sat pump "   << std::endl;
		  if (excess.get(w)==0) break;
		  /*If w is not active any more, then we go on to the next Node.*/
		  
		} // if (capacity.get(e)-flow.get(e) > excess.get(w))
	      } // if(level.get(w)==level.get(v)+1)
	    
	      else {newlevel = newlevel < level.get(v) ? newlevel : level.get(v);}
	    
	    } //if (flow.get(e)<capacity.get(e))
	 
	  } //for(OutEdgeIt e=G.first_OutEdge(w); e.valid(); ++e) 
	  


	  for(InEdgeIt e=G.template first<InEdgeIt>(w); e.valid(); ++e) {
	    NodeIt v=G.tail(e);
	    /*e is the Edge vw.*/

	    if (excess.get(w)==0) break;
	    /*It may happen, that w became inactive in the first for cycle.*/		
	    if(flow.get(e)>0) {             
	      /*e is an Edge of the residual graph */

	      if(level.get(w)==level.get(v)+1) {  
		/*Push is allowed now*/
		
		if (flow.get(e) > excess.get(w)) { 
		  /*A nonsaturating push.*/
		  
		  if (excess.get(v)==0 && v != s) stack[level.get(v)].push(v); 
		  /*v becomes active.*/

		  flow.set(e, flow.get(e)-excess.get(w));
		  excess.set(v, excess.get(v)+excess.get(w));
		  excess.set(w,0);
		  //std::cout << v << " " << w << " vissza elen nonsat pump "     << std::endl;
		  break; 
		} else {                                               
		  /*A saturating push.*/
		  
		  if (excess.get(v)==0 && v != s) stack[level.get(v)].push(v); 
		  /*v becomes active.*/
		  
		  excess.set(v, excess.get(v)+flow.get(e));
		  excess.set(w, excess.get(w)-flow.get(e));
		  flow.set(e,0);
		  //std::cout << v <<" " << w << " vissza elen sat pump "     << std::endl;
		  if (excess.get(w)==0) { break;}
		} //if (flow.get(e) > excess.get(v)) 
	      } //if(level.get(w)==level.get(v)+1)
	      
	      else {newlevel = newlevel < level.get(v) ? newlevel : level.get(v);}
	      

	    } //if (flow.get(e)>0)

	  } //for


	  if (excess.get(w)>0) {
	    level.set(w,++newlevel);
	    stack[newlevel].push(w);
	    b=newlevel;
	    //std::cout << "The new level of " << w << " is "<< newlevel <<std::endl; 
	  }


	} //else
       
      } //while(b)

      value = excess.get(t);
      /*Max flow value.*/




    } //void run()





    /*
      Returns the maximum value of a flow.
     */

    T maxflow() {
      return value;
    }



    /*
      For the maximum flow x found by the algorithm, it returns the flow value on Edge e, i.e. x(e). 
    */

    T flowonEdge(EdgeIt e) {
      return flow.get(e);
    }



    /*
      Returns the maximum flow x found by the algorithm.
    */

    EdgeMap<graph_type, T> allflow() {
      return flow;
    }



    /*
      Returns a minimum cut by using a reverse bfs from t in the residual graph.
    */
    
    NodeMap<graph_type, bool> mincut() {
    
      std::queue<NodeIt> queue;
      
      mincutvector.set(t,false);      
      queue.push(t);

      while (!queue.empty()) {
        NodeIt w=queue.front();
	queue.pop();

	for(InEdgeIt e=G.template first<InEdgeIt>(w) ; e.valid(); ++e) {
	  NodeIt v=G.tail(e);
	  if (mincutvector.get(v) && flow.get(e) < capacity.get(e) ) {
	    queue.push(v);
	    mincutvector.set(v, false);
	  }
	} // for

	for(OutEdgeIt e=G.template first<OutEdgeIt>(w) ; e.valid(); ++e) {
	  NodeIt v=G.head(e);
	  if (mincutvector.get(v) && flow.get(e) > 0 ) {
	    queue.push(v);
	    mincutvector.set(v, false);
	  }
	} // for

      }

      return mincutvector;
    
    }


  };
}//namespace marci
#endif 




