/* -*- C++ -*-
 * src/hugo/map_defines.h - Part of HUGOlib, a generic C++ optimization library
 *
 * Copyright (C) 2004 Egervary Jeno Kombinatorikus Optimalizalasi Kutatocsoport
 * (Egervary Combinatorial Optimization Research Group, EGRES).
 *
 * Permission to use, modify and distribute this software is granted
 * provided that this copyright notice appears in all copies. For
 * precise terms see the accompanying LICENSE file.
 *
 * This software is provided "AS IS" with no warranty of any kind,
 * express or implied, and with no claim as to its suitability for any
 * purpose.
 *
 */

#ifndef HUGO_MAP_DEFINES_H
#define HUGO_MAP_DEFINES_H

///\ingroup graphmaps
///\file
///\brief Defines to help creating graph maps.

/// \addtogroup graphmapfactory
/// @{

/** Creates the EdgeMapRegistry type an declare a mutable instance 
 *  named edge_maps.
 */
#define CREATE_EDGE_MAP_REGISTRY \
typedef MapRegistry<Graph, Edge, EdgeIt> EdgeMapRegistry; \
mutable EdgeMapRegistry edge_maps;

/** Creates the NodeMapRegistry type an declare a mutable instance 
 *  named node_maps.
 */
#define CREATE_NODE_MAP_REGISTRY \
typedef MapRegistry<Graph, Node, NodeIt> NodeMapRegistry; \
mutable NodeMapRegistry node_maps;

/** Creates both map registries.
 */
#define CREATE_MAP_REGISTRIES \
CREATE_NODE_MAP_REGISTRY \
CREATE_EDGE_MAP_REGISTRY

/** Creates a map from a template map. The import method is
 *  an overloading of the map type.
 *  The reason to use these macro is that the c++ does not support
 *  the template typedefs. If a future release of the c++ 
 *  supports this feature it should be fixed.
 */
#define CREATE_NODE_MAP(DynMap) \
template <typename Value> \
class NodeMap : public DynMap<NodeMapRegistry, Value> { \
public: \
typedef DynMap<NodeMapRegistry, Value> Parent; \
NodeMap(const typename Parent::Graph& g) \
  : Parent(g, g.node_maps) {} \
NodeMap(const typename Parent::Graph& g, const Value& v) \
  : Parent(g, g.node_maps, v) {} \
NodeMap(const NodeMap& copy) : Parent(static_cast<const Parent&>(copy)) {} \
template <typename TT> \
NodeMap(const NodeMap<TT>& copy) \
  : Parent(static_cast<const typename NodeMap<TT>::Parent&>(copy)) {} \
NodeMap& operator=(const NodeMap& copy) { \
  Parent::operator=(static_cast<const Parent&>(copy));\
  return *this; \
} \
template <typename TT> \
NodeMap& operator=(const NodeMap<TT>& copy) { \
  Parent::operator=(static_cast<const typename NodeMap<TT>::Parent&>(copy));\
  return *this; \
} \
};

/** Creates a map from a template map. The import method is
 *  an overloading of the map type.
 *  The reason to use these macro is that the c++ does not support
 *  the template typedefs. If a future release of the c++ 
 *  supports this feature it should be fixed.
 */
#define CREATE_EDGE_MAP(DynMap) \
template <typename Value> \
class EdgeMap : public DynMap<EdgeMapRegistry, Value> { \
public: \
typedef DynMap<EdgeMapRegistry, Value> Parent; \
\
EdgeMap(const typename Parent::Graph& g) \
  : Parent(g, g.edge_maps) {} \
EdgeMap(const typename Parent::Graph& g, const Value& v) \
  : Parent(g, g.edge_maps, v) {} \
EdgeMap(const EdgeMap& copy) : Parent(static_cast<const Parent&>(copy)) {} \
template <typename TT> \
EdgeMap(const EdgeMap<TT>& copy) \
  : Parent(static_cast<const typename EdgeMap<TT>::Parent&>(copy)) {} \
EdgeMap& operator=(const EdgeMap& copy) { \
  Parent::operator=(static_cast<const Parent&>(copy));\
  return *this; \
} \
template <typename TT> \
EdgeMap& operator=(const EdgeMap<TT>& copy) { \
  Parent::operator=(static_cast<const typename EdgeMap<TT>::Parent&>(copy));\
  return *this; \
} \
};

/** This macro creates both maps.
 */
#define CREATE_MAPS(DynMap) \
CREATE_NODE_MAP(DynMap) \
CREATE_EDGE_MAP(DynMap)

/** This macro creates MapRegistry for Symmetric Edge Maps.
 */
#define CREATE_SYM_EDGE_MAP_REGISTRY \
typedef SymEdgeIt<Graph, Edge, EdgeIt> SymEdgeIt; \
typedef MapRegistry<Graph, Edge, SymEdgeIt> SymEdgeMapRegistry; \
mutable SymEdgeMapRegistry sym_edge_maps;


/** Creates a map from a template map. The import method is
 *  an overloading of the map type.
 *  The reason to use these macro is that the c++ does not support
 *  the template typedefs. If a future release of the c++ 
 *  supports this feature it should be fixed.
 */
#define CREATE_SYM_EDGE_MAP(DynMap) \
template <typename Value> \
class SymEdgeMap : public SymMap<DynMap, SymEdgeMapRegistry, Value> { \
public: \
typedef SymMap<DynMap, SymEdgeMapRegistry, Value> Parent; \
\
SymEdgeMap(const typename Parent::Graph& g) \
  : Parent(g, g.sym_edge_maps) {} \
SymEdgeMap(const typename Parent::Graph& g, const Value& v) \
  : Parent(g, g.sym_edge_maps, v) {} \
SymEdgeMap(const SymEdgeMap& copy) \
  : Parent(static_cast<const Parent&>(copy)) {} \
template <typename TT> \
SymEdgeMap(const NodeMap<TT>& copy) \
  : Parent(static_cast<const typename SymEdgeMap<TT>::Parent&>(copy)) {} \
SymEdgeMap& operator=(const SymEdgeMap& copy) { \
  Parent::operator=(static_cast<const Parent&>(copy));\
  return *this; \
} \
template <typename TT> \
SymEdgeMap& operator=(const SymEdgeMap<TT>& copy) { \
  Parent::operator=(static_cast<const typename SymEdgeMap<TT>::Parent&>(copy));\
  return *this; \
} \
};

/** This is a macro to import an node map into a graph class.
 */
#define IMPORT_NODE_MAP(From, from, To, to) \
template <typename Value> \
class NodeMap : public From::template NodeMap<Value> { \
\
public: \
typedef typename From::template NodeMap<Value> Parent; \
\
NodeMap(const To& to) \
  : Parent(static_cast<const From&>(from)) { } \
NodeMap(const To& to, const Value& value) \
  : Parent(static_cast<const From&>(from), value) { } \
NodeMap(const NodeMap& copy) \
  : Parent(static_cast<const Parent&>(copy)) {} \
template <typename TT> \
NodeMap(const NodeMap<TT>& copy) \
  : Parent(static_cast<const typename NodeMap<TT>::Parent&>(copy)) {} \
NodeMap& operator=(const NodeMap& copy) { \
  Parent::operator=(static_cast<const Parent&>(copy)); \
  return *this; \
} \
template <typename TT> \
NodeMap& operator=(const NodeMap<TT>& copy) { \
  Parent::operator=(static_cast<const typename NodeMap<TT>::Parent&>(copy));\
  return *this; \
} \
};

/** This is a macro to import an edge map into a graph class.
 */
#define IMPORT_EDGE_MAP(From, from, To, to) \
template <typename Value> \
class EdgeMap : public From::template EdgeMap<Value> { \
\
public: \
typedef typename From::template EdgeMap<Value> Parent; \
\
EdgeMap(const To& to) \
  : Parent(static_cast<const From&>(from)) { } \
EdgeMap(const To& to, const Value& value) \
  : Parent(static_cast<const From&>(from), value) { } \
EdgeMap(const EdgeMap& copy) \
  : Parent(static_cast<const Parent&>(copy)) {} \
template <typename TT> \
EdgeMap(const EdgeMap<TT>& copy) \
  : Parent(static_cast<const typename EdgeMap<TT>::Parent&>(copy)) {} \
EdgeMap& operator=(const EdgeMap& copy) { \
  Parent::operator=(static_cast<const Parent&>(copy)); \
  return *this; \
} \
template <typename TT> \
EdgeMap& operator=(const EdgeMap<TT>& copy) { \
  Parent::operator=(static_cast<const typename EdgeMap<TT>::Parent&>(copy));\
  return *this; \
} \
};

#define KEEP_EDGE_MAP(From, To) \
IMPORT_EDGE_MAP(From, graph, To, graph)


#define KEEP_NODE_MAP(From, To) \
IMPORT_NODE_MAP(From, graph, To, graph)

/** This is a macro to keep the node and edge maps for a graph class.
 */
#define KEEP_MAPS(From, To) \
KEEP_EDGE_MAP(From, To) \
KEEP_NODE_MAP(From, To)

  
/// @}
  
#endif
