// -*- mode:C++ -*-

#ifndef HUGO_LIST_GRAPH_H
#define HUGO_LIST_GRAPH_H

///\ingroup graphs
///\file
///\brief ListGraph, SymListGraph, NodeSet and EdgeSet classes.

#include <vector>
#include <climits>

#include "invalid.h"

#include "array_map_factory.h"
#include "map_registry.h"

#include "map_defines.h"

namespace hugo {

/// \addtogroup graphs
/// @{

  ///A list graph class.

  ///This is a simple and fast erasable graph implementation.
  ///
  ///It conforms to the graph interface documented under
  ///the description of \ref Graph.
  ///\sa \ref Graph.
  class ListGraph {

    //Nodes are double linked.
    //The free nodes are only single linked using the "next" field.
    struct NodeT 
    {
      int first_in,first_out;
      int prev, next;
      //      NodeT() {}
    };
    //Edges are double linked.
    //The free edges are only single linked using the "next_in" field.
    struct EdgeT 
    {
      int head, tail;
      int prev_in, prev_out;
      int next_in, next_out;
      //FIXME: is this necessary?
      //      EdgeT() : next_in(-1), next_out(-1) prev_in(-1), prev_out(-1) {}  
    };

    std::vector<NodeT> nodes;
    //The first node
    int first_node;
    //The first free node
    int first_free_node;
    std::vector<EdgeT> edges;
    //The first free edge
    int first_free_edge;
    
  protected:
    
  public:
    
    class Node;
    class Edge;

    typedef ListGraph Graph;

  public:

    class NodeIt;
    class EdgeIt;
    class OutEdgeIt;
    class InEdgeIt;
    
    CREATE_MAP_REGISTRIES;
    CREATE_MAPS(ArrayMapFactory);
  public:

    ListGraph() : nodes(), first_node(-1),
		  first_free_node(-1), edges(), first_free_edge(-1) {}
    ListGraph(const ListGraph &_g) : nodes(_g.nodes), first_node(_g.first_node),
				     first_free_node(_g.first_free_node),
				     edges(_g.edges),
				     first_free_edge(_g.first_free_edge) {}
    

    int nodeNum() const { return nodes.size(); }  //FIXME: What is this?
    int edgeNum() const { return edges.size(); }  //FIXME: What is this?

    ///Set the expected number of edges

    ///With this function, it is possible to set the expected number of edges.
    ///The use of this fasten the building of the graph and makes
    ///it possible to avoid the superfluous memory allocation.
    void reserveEdge(int n) { edges.reserve(n); };
    
    ///\bug This function does something different than
    ///its name would suggests...
    int maxNodeId() const { return nodes.size(); }  //FIXME: What is this?
    ///\bug This function does something different than
    ///its name would suggests...
    int maxEdgeId() const { return edges.size(); }  //FIXME: What is this?

    Node tail(Edge e) const { return edges[e.n].tail; }
    Node head(Edge e) const { return edges[e.n].head; }

    Node aNode(OutEdgeIt e) const { return edges[e.n].tail; }
    Node aNode(InEdgeIt e) const { return edges[e.n].head; }

    Node bNode(OutEdgeIt e) const { return edges[e.n].head; }
    Node bNode(InEdgeIt e) const { return edges[e.n].tail; }

    NodeIt& first(NodeIt& v) const { 
      v=NodeIt(*this); return v; }
    EdgeIt& first(EdgeIt& e) const { 
      e=EdgeIt(*this); return e; }
    OutEdgeIt& first(OutEdgeIt& e, const Node v) const { 
      e=OutEdgeIt(*this,v); return e; }
    InEdgeIt& first(InEdgeIt& e, const Node v) const { 
      e=InEdgeIt(*this,v); return e; }

//     template< typename It >
//     It first() const { It e; first(e); return e; }

//     template< typename It >
//     It first(Node v) const { It e; first(e,v); return e; }

    bool valid(Edge e) const { return e.n!=-1; }
    bool valid(Node n) const { return n.n!=-1; }
    
    void setInvalid(Edge &e) { e.n=-1; }
    void setInvalid(Node &n) { n.n=-1; }
    
    template <typename It> It getNext(It it) const
    { It tmp(it); return next(tmp); }

    NodeIt& next(NodeIt& it) const { 
      it.n=nodes[it.n].next; 
      return it; 
    }
    OutEdgeIt& next(OutEdgeIt& it) const
    { it.n=edges[it.n].next_out; return it; }
    InEdgeIt& next(InEdgeIt& it) const
    { it.n=edges[it.n].next_in; return it; }
    EdgeIt& next(EdgeIt& it) const {
      if(edges[it.n].next_in!=-1) { 
	it.n=edges[it.n].next_in;
      }
      else {
	int n;
	for(n=nodes[edges[it.n].head].next;
	    n!=-1 && nodes[n].first_in == -1;
	    n = nodes[n].next) ;
	it.n = (n==-1)?-1:nodes[n].first_in;
      }
      return it;
    }

    int id(Node v) const { return v.n; }
    int id(Edge e) const { return e.n; }

    /// Adds a new node to the graph.

    /// \todo It adds the nodes in a reversed order.
    /// (i.e. the lastly added node becomes the first.)
    Node addNode() {
      int n;
      
      if(first_free_node==-1)
	{
	  n = nodes.size();
	  nodes.push_back(NodeT());
	}
      else {
	n = first_free_node;
	first_free_node = nodes[n].next;
      }
      
      nodes[n].next = first_node;
      if(first_node != -1) nodes[first_node].prev = n;
      first_node = n;
      nodes[n].prev = -1;
      
      nodes[n].first_in = nodes[n].first_out = -1;
      
      Node nn; nn.n=n;

      //Update dynamic maps
      node_maps.add(nn);

      return nn;
    }
    
    Edge addEdge(Node u, Node v) {
      int n;
      
      if(first_free_edge==-1)
	{
	  n = edges.size();
	  edges.push_back(EdgeT());
	}
      else {
	n = first_free_edge;
	first_free_edge = edges[n].next_in;
      }
      
      edges[n].tail = u.n; edges[n].head = v.n;

      edges[n].next_out = nodes[u.n].first_out;
      if(nodes[u.n].first_out != -1) edges[nodes[u.n].first_out].prev_out = n;
      edges[n].next_in = nodes[v.n].first_in;
      if(nodes[v.n].first_in != -1) edges[nodes[v.n].first_in].prev_in = n;
      edges[n].prev_in = edges[n].prev_out = -1;
	
      nodes[u.n].first_out = nodes[v.n].first_in = n;

      Edge e; e.n=n;

      //Update dynamic maps
      edge_maps.add(e);

      return e;
    }

  private:
    void eraseEdge(int n) {
      
      if(edges[n].next_in!=-1)
	edges[edges[n].next_in].prev_in = edges[n].prev_in;
      if(edges[n].prev_in!=-1)
	edges[edges[n].prev_in].next_in = edges[n].next_in;
      else nodes[edges[n].head].first_in = edges[n].next_in;
      
      if(edges[n].next_out!=-1)
	edges[edges[n].next_out].prev_out = edges[n].prev_out;
      if(edges[n].prev_out!=-1)
	edges[edges[n].prev_out].next_out = edges[n].next_out;
      else nodes[edges[n].tail].first_out = edges[n].next_out;
      
      edges[n].next_in = first_free_edge;
      first_free_edge = n;      

      //Update dynamic maps
      Edge e; e.n=n;
    }
      
  public:

    void erase(Node nn) {
      int n=nn.n;
      
      int m;
      while((m=nodes[n].first_in)!=-1) eraseEdge(m);
      while((m=nodes[n].first_out)!=-1) eraseEdge(m);

      if(nodes[n].next != -1) nodes[nodes[n].next].prev = nodes[n].prev;
      if(nodes[n].prev != -1) nodes[nodes[n].prev].next = nodes[n].next;
      else first_node = nodes[n].next;
      
      nodes[n].next = first_free_node;
      first_free_node = n;

      //Update dynamic maps
      node_maps.erase(nn);
     }
    
    void erase(Edge e) { 
      edge_maps.erase(e);
      eraseEdge(e.n); 
    }

    ///\bug Dynamic maps must be updated!
    ///
    void clear() {
      nodes.clear();edges.clear();
      first_node=first_free_node=first_free_edge=-1;
    }

    class Node {
      friend class ListGraph;
      template <typename T> friend class NodeMap;
       
      friend class Edge;
      friend class OutEdgeIt;
      friend class InEdgeIt;
      friend class SymEdge;

    protected:
      int n;
      friend int ListGraph::id(Node v) const; 
      Node(int nn) {n=nn;}
    public:
      Node() {}
      Node (Invalid) { n=-1; }
      bool operator==(const Node i) const {return n==i.n;}
      bool operator!=(const Node i) const {return n!=i.n;}
      bool operator<(const Node i) const {return n<i.n;}
    };
    
    class NodeIt : public Node {
      friend class ListGraph;
    public:
      NodeIt() : Node() { }
      NodeIt(Invalid i) : Node(i) { }
      NodeIt(const ListGraph& G) : Node(G.first_node) { }
      ///\todo Undocumented conversion Node -\> NodeIt.
      NodeIt(const ListGraph& G, const Node &n) : Node(n) { }
    };

    class Edge {
      friend class ListGraph;
      template <typename T> friend class EdgeMap;

      //template <typename T> friend class SymListGraph::SymEdgeMap;      
      //friend Edge SymListGraph::opposite(Edge) const;
      
      friend class Node;
      friend class NodeIt;
    protected:
      int n;
      friend int ListGraph::id(Edge e) const;

      Edge(int nn) {n=nn;}
    public:
      Edge() { }
      Edge (Invalid) { n=-1; }
      bool operator==(const Edge i) const {return n==i.n;}
      bool operator!=(const Edge i) const {return n!=i.n;}
      bool operator<(const Edge i) const {return n<i.n;}
      ///\bug This is a workaround until somebody tells me how to
      ///make class \c SymListGraph::SymEdgeMap friend of Edge
      int &idref() {return n;}
      const int &idref() const {return n;}
    };
    
    class EdgeIt : public Edge {
      friend class ListGraph;
    public:
      EdgeIt(const ListGraph& G) : Edge() {
      	int m;
	for(m=G.first_node;
	    m!=-1 && G.nodes[m].first_in == -1; m = G.nodes[m].next);
	n = (m==-1)?-1:G.nodes[m].first_in;
      }
      EdgeIt (Invalid i) : Edge(i) { }
      EdgeIt() : Edge() { }
      ///\bug This is a workaround until somebody tells me how to
      ///make class \c SymListGraph::SymEdgeMap friend of Edge
      int &idref() {return n;}
    };
    
    class OutEdgeIt : public Edge {
      friend class ListGraph;
    public: 
      OutEdgeIt() : Edge() { }
      OutEdgeIt (Invalid i) : Edge(i) { }

      OutEdgeIt(const ListGraph& G,const Node v)
	: Edge(G.nodes[v.n].first_out) {}
    };
    
    class InEdgeIt : public Edge {
      friend class ListGraph;
    public: 
      InEdgeIt() : Edge() { }
      InEdgeIt (Invalid i) : Edge(i) { }
      InEdgeIt(const ListGraph& G,Node v) :Edge(G.nodes[v.n].first_in) {}
    };

  };

  ///Graph for bidirectional edges.

  ///The purpose of this graph structure is to handle graphs
  ///having bidirectional edges. Here the function \c addEdge(u,v) adds a pair
  ///of oppositely directed edges.
  ///There is a new edge map type called
  ///\ref SymListGraph::SymEdgeMap "SymEdgeMap"
  ///that complements this
  ///feature by
  ///storing shared values for the edge pairs. The usual
  ///\ref Graph::EdgeMap "EdgeMap"
  ///can be used
  ///as well.
  ///
  ///The oppositely directed edge can also be obtained easily
  ///using \ref opposite.
  ///
  ///Here erase(Edge) deletes a pair of edges.
  ///
  ///\todo this date structure need some reconsiderations. Maybe it
  ///should be implemented independently from ListGraph.

}

#endif //HUGO_LIST_GRAPH_H
