#ifndef EDGE_MAP_BASE_H
#define EDGE_MAP_BASE_H

/**
	Template base class for implementing mapping on edges.
	\param The first template parameter is the Graph class. The Graph
		must have an \emp edge_maps member with \emp EdgeMapRegistry class.
	\param The second template parameter is the Edge type of the class.
	
*/

template <typename G, typename K>
class EdgeMapBase {
public:
	typedef G Graph;

	typedef K KeyType;
	
	/** 
		Default constructor.
	*/	
	
	EdgeMapBase() : graph(0) {}

	/** 
		Simple constructor to register into a graph.
	*/
	
	EdgeMapBase(Graph& g) : graph(&g) {
		graph->edge_maps.add(*this);
	}

	/** 
		Copy constructor with registering into the map.
	*/	
	
	EdgeMapBase(const EdgeMapBase& copy) : graph(copy.graph) {
		if (graph) {
			graph->edge_maps.add(*this);
		}
	} 
	
	/** 
		Assign operator.
	*/	

	const EdgeMapBase& operator=(const EdgeMapBase& copy) {
		if (graph) {
			graph.edge_maps.erase(*this);
		}
		graph = copy.graph;
		if (graph) {
			graph->edge_maps.add(*this);
		}
		
	}
	

	/** 
		Destructor.
	*/	

	virtual ~EdgeMapBase() {
		if (graph) {
			graph.edge_maps.erase(*this);
		}
	}
	
protected:
	
	Graph* graph;

	int graph_index;
	
	/**
		Helper function to implement the default constructor in the subclasses.
	*/
	
	void init() {
		for (Graph::EdgeIt it(g); g.valid(it); g.next(it)) {
			add(it);
		}
	}
	
	/**
		Helper function to implement the destructor in the subclasses.
	*/
	
	void destroy() {
		for (Graph::EdgeIt it(g); g.valid(it); g.next(it)) {
			erase(it);
		}
	}
	
	/** 
		The add member function should be overloaded in the subclasses.
		\e Add extends the map with the new edge.
	*/
	
	virtual void add(const KeyType&) = 0;
	
	/** 
		The erase member function should be overloaded in the subclasses.
		\e Erase removes the edge from the map.
	*/
	
	virtual void erase(const KeyType&) = 0;
	
	/**
		Exception class to throw at unsupported operation.
	*/
	
	class NotSupportedOperationException {};

	friend class Graph;
};

#endif
