/* -*- C++ -*-
 * src/lemon/bezier.h - Part of LEMON, a generic C++ optimization library
 *
 * Copyright (C) 2004 Egervary Jeno Kombinatorikus Optimalizalasi Kutatocsoport
 * (Egervary Combinatorial Optimization Research Group, EGRES).
 *
 * Permission to use, modify and distribute this software is granted
 * provided that this copyright notice appears in all copies. For
 * precise terms see the accompanying LICENSE file.
 *
 * This software is provided "AS IS" with no warranty of any kind,
 * express or implied, and with no claim as to its suitability for any
 * purpose.
 *
 */

#ifndef LEMON_BEZIER_H
#define LEMON_BEZIER_H

///\ingroup misc
///\file
///\brief Classes to compute with Bezier curves.
///
///Up to now this file is internally used by \ref graph_to_eps.h
///
///\author Alpar Juttner

#include<lemon/xy.h>

namespace lemon {

class BezierBase {
public:
  typedef xy<double> xy;
protected:
  static xy conv(xy x,xy y,double t) {return (1-t)*x+t*y;}
};

class Bezier1 : public BezierBase
{
public:
  xy p1,p2;

  Bezier1() {}
  Bezier1(xy _p1, xy _p2) :p1(_p1), p2(_p2) {}
  
  xy operator()(double t) const
  {
    //    return conv(conv(p1,p2,t),conv(p2,p3,t),t);
    return conv(p1,p2,t);
  }
  Bezier1 before(double t) const
  {
    return Bezier1(p1,conv(p1,p2,t));
  }
  
  Bezier1 after(double t) const
  {
    return Bezier1(conv(p1,p2,t),p2);
  }
  Bezier1 operator()(double a,double b) { return before(b).after(a/b); }  
};

class Bezier2 : public BezierBase
{
public:
  xy p1,p2,p3;

  Bezier2() {}
  Bezier2(xy _p1, xy _p2, xy _p3) :p1(_p1), p2(_p2), p3(_p3) {}
  Bezier2(const Bezier1 &b) : p1(b.p1), p2(conv(b.p1,b.p2,.5)), p3(b.p2) {}
  xy operator()(double t) const
  {
    //    return conv(conv(p1,p2,t),conv(p2,p3,t),t);
    return ((1-t)*(1-t))*p1+(2*(1-t)*t)*p2+(t*t)*p3;
  }
  Bezier2 before(double t) const
  {
    xy q(conv(p1,p2,t));
    xy r(conv(p2,p3,t));
    return Bezier2(p1,q,conv(q,r,t));
  }
  
  Bezier2 after(double t) const
  {
    xy q(conv(p1,p2,t));
    xy r(conv(p2,p3,t));
    return Bezier2(conv(q,r,t),r,p3);
  }
  Bezier2 operator()(double a,double b) { return before(b).after(a/b); }
  
};

class Bezier3 : public BezierBase
{
public:
  xy p1,p2,p3,p4;

  Bezier3() {}
  Bezier3(xy _p1, xy _p2, xy _p3, xy _p4) :p1(_p1), p2(_p2), p3(_p3), p4(_p4) {}
  Bezier3(const Bezier1 &b) : p1(b.p1), p2(conv(b.p1,b.p2,1.0/3.0)), 
			      p3(conv(b.p1,b.p2,2.0/3.0)), p4(b.p2) {}
  Bezier3(const Bezier2 &b) : p1(b.p1), p2(conv(b.p1,b.p2,2.0/3.0)),
			      p3(conv(b.p2,b.p3,1.0/3.0)), p4(b.p3) {}
  
  xy operator()(double t) const 
    {
      //    return Bezier2(conv(p1,p2,t),conv(p2,p3,t),conv(p3,p4,t))(t);
      return ((1-t)*(1-t)*(1-t))*p1+(3*t*(1-t)*(1-t))*p2+
	(3*t*t*(1-t))*p3+(t*t*t)*p4;
    }
  Bezier3 before(double t) const
    {
      xy p(conv(p1,p2,t));
      xy q(conv(p2,p3,t));
      xy r(conv(p3,p4,t));
      xy a(conv(p,q,t));
      xy b(conv(q,r,t));
      xy c(conv(a,b,t));
      return Bezier3(p1,p,a,c);
    }
  
  Bezier3 after(double t) const
    {
      xy p(conv(p1,p2,t));
      xy q(conv(p2,p3,t));
      xy r(conv(p3,p4,t));
      xy a(conv(p,q,t));
      xy b(conv(q,r,t));
      xy c(conv(a,b,t));
      return Bezier3(c,b,r,p4);
    }
  Bezier3 operator()(double a,double b) { return before(b).after(a/b); }
  
};

} //END OF NAMESPACE LEMON

#endif // LEMON_BEZIER_H
