/* -*- C++ -*-
 * src/test/max_matching_test.cc - Part of LEMON, a generic C++ optimization library
 *
 * Copyright (C) 2004 Egervary Jeno Kombinatorikus Optimalizalasi Kutatocsoport
 * (Egervary Combinatorial Optimization Research Group, EGRES).
 *
 * Permission to use, modify and distribute this software is granted
 * provided that this copyright notice appears in all copies. For
 * precise terms see the accompanying LICENSE file.
 *
 * This software is provided "AS IS" with no warranty of any kind,
 * express or implied, and with no claim as to its suitability for any
 * purpose.
 *
 */

#include <iostream>
#include <queue>
#include <math.h>
#include <cstdlib>

#include "test_tools.h"
#include <lemon/invalid.h>
#include <lemon/list_graph.h>
#include <graph_gen.h>
#include <max_matching.h>

using namespace std;
using namespace lemon;

int main() {

  typedef UndirListGraph Graph;

  typedef Graph::Edge Edge;
  typedef Graph::UndirEdgeIt UndirEdgeIt;
  typedef Graph::IncEdgeIt IncEdgeIt;
  typedef Graph::NodeIt NodeIt;
  typedef Graph::Node Node;
   
  Graph G;

  random_init(); 
  randomGraph(G, random(5000), random(20000) );  

  MaxMatching<Graph> max_matching(G);
  max_matching.runEdmonds(0);
  
  int s=0;
  Graph::NodeMap<Node> mate(G,INVALID);
  max_matching.writeNMapNode(mate);
  for(NodeIt v(G); v!=INVALID; ++v) {
    if ( mate[v]!=INVALID ) ++s;
  }
  int size=(int)s/2;  //size will be used as the size of a maxmatching
  
  check ( size == max_matching.size(), "mate() returns a different size matching than max_matching.size()" );

  Graph::NodeMap<MaxMatching<Graph>::pos_enum> pos0(G);
  max_matching.writePos(pos0);
  
  max_matching.resetPos();
  max_matching.resetMatching();
  max_matching.runEdmonds(1);
  s=0;
  max_matching.writeNMapNode(mate);
  for(NodeIt v(G); v!=INVALID; ++v) {
    if ( mate[v]!=INVALID ) ++s;
  }
  check ( (int)s/2 == size, "The size does not equal!" );

  Graph::NodeMap<MaxMatching<Graph>::pos_enum> pos1(G);
  max_matching.writePos(pos1);

  max_matching.resetPos();
  max_matching.run();
  s=0;
  max_matching.writeNMapNode(mate);
  for(NodeIt v(G); v!=INVALID; ++v) {
    if ( mate[v]!=INVALID ) ++s;
  }
  check ( (int)s/2 == size, "The size does not equal!" ); 
  
  Graph::NodeMap<MaxMatching<Graph>::pos_enum> pos2(G);
  max_matching.writePos(pos2);

  max_matching.resetPos();
  max_matching.resetMatching();
  max_matching.run();
  s=0;
  max_matching.writeNMapNode(mate);
  for(NodeIt v(G); v!=INVALID; ++v) {
    if ( mate[v]!=INVALID ) ++s;
  }
  check ( (int)s/2 == size, "The size does not equal!" ); 
  
  Graph::NodeMap<MaxMatching<Graph>::pos_enum> pos(G);
  max_matching.writePos(pos);
   
  bool ismatching=true;
  for(NodeIt v(G); v!=INVALID; ++v) {
    if ( mate[v]!=INVALID ) {
      Node u=mate[v];
      if (mate[u]!=v) ismatching=false; 
    }
  }  
  check ( ismatching, "It is not a matching!" );

  bool coincide=true;
  for(NodeIt v(G); v!=INVALID; ++v) {
   if ( pos0[v] != pos1[v] || pos1[v]!=pos2[v] || pos2[v]!=pos[v] ) {
     coincide=false;
    }
  }
  check ( coincide, "The decompositions do not coincide! " );

  bool noedge=true;
  for(UndirEdgeIt e(G); e!=INVALID; ++e) {
   if ( (pos[G.target(e)]==max_matching.C && pos[G.source(e)]==max_matching.D) || 
	 (pos[G.target(e)]==max_matching.D && pos[G.source(e)]==max_matching.C) )
      noedge=false; 
  }
  check ( noedge, "There are edges between D and C!" );
  
  bool oddcomp=true;
  Graph::NodeMap<bool> todo(G,true);
  int num_comp=0;
  for(NodeIt v(G); v!=INVALID; ++v) {
   if ( pos[v]==max_matching.D && todo[v] ) {
      int comp_size=1;
      ++num_comp;
      std::queue<Node> Q;
      Q.push(v);
      todo.set(v,false);
      while (!Q.empty()) {
	Node w=Q.front();	
	Q.pop();
	for(IncEdgeIt e(G,w); e!=INVALID; ++e) {
	  Node u=G.target(e);
	  if ( pos[u]==max_matching.D && todo[u] ) {
	    ++comp_size;
	    Q.push(u);
	    todo.set(u,false);
	  }
	}
      }
      if ( !(comp_size % 2) ) oddcomp=false;  
    }
  }
  check ( oddcomp, "A component of G[D] is not odd." );

  int barrier=0;
  for(NodeIt v(G); v!=INVALID; ++v) {
    if ( pos[v]==max_matching.A ) ++barrier;
  }
  int expected_size=(int)( countNodes(G)-num_comp+barrier)/2;
  check ( size==expected_size, "The size of the matching is wrong." );
  
  return 0;
}


void random_init()
{
  unsigned int seed = getpid();
  seed |= seed << 15;
  seed ^= time(0);
  
  srand(seed);
}


int random(int m)
{
  return int( double(m) * rand() / (RAND_MAX + 1.0) );
}


/// Generates a random graph with n nodes and m edges.
/// Before generating the random graph, \c g.clear() is called.
template<typename Graph>
void randomGraph(Graph& g, int n, int m) {
  g.clear();
  std::vector<typename Graph::Node> nodes;
  for (int i=0; i<n; ++i)
    nodes.push_back(g.addNode());
  for (int i=0; i<m; ++i) 
    g.addEdge(nodes[random(n)], nodes[random(n)]);
}
