#ifndef MAP_BASE_H
#define MAP_BASE_H

/**
	Template base class for implementing mapping on nodes.
	\param The first template parameter is the Graph class. The Graph
		must have an \emp node_maps member with \emp MapRegistry class.
	\param The second template parameter is the  type of the class.
	
*/


namespace hugo {
	template <typename G, typename K, typename KIt>
	class MapBase;
}

#include "map_registry.h"

namespace hugo {

	template <typename G, typename K, typename KIt>
	class MapBase {
	public:
		typedef G Graph;
		typedef MapRegistry<G, K, KIt> Registry;
		typedef K KeyType;
		typedef KIt KeyIt;
	
		friend class Registry;
		
		/** 
			Default constructor.
		*/	
		
		MapBase() : registry(0) {}

		/** 
			Simple constructor to register into a graph registry.
		*/
	
		MapBase(Graph& g, Registry& r) : graph(&g), registry(0) {
			registry->attach(*this);
		}

		/** 
			Copy constructor with registering into the map.
		*/	
	
		MapBase(const MapBase& copy) : registry(0), graph(copy.graph) {
			if (copy.registry) {
				copy.registry->attach(*this);
			}
		} 
	
		/** 
			Assign operator.
		*/	

		const MapBase& operator=(const MapBase& copy) {
			if (registry) {
				registry->detach(*this);
			}
			graph = copy.graph;
			if (copy.registry) {
				copy.registry->attach(*this);
			}
		}
	

		/** 
			Destructor.
		*/		

		virtual ~MapBase() {
			if (registry) {
				registry->detach(*this);
			}
		}
	
	protected:
		
		Registry* registry;
		Graph* graph;

		int registry_index;
	
		/**
			Helper function to implement the default constructor in the subclasses.
		*/
	
		virtual void init() {

			for (KeyIt it(*graph); graph->valid(it); graph->next(it)) {
				add(it);
			}
		}
	
		/**
			Helper function to implement the destructor in the subclasses.
		*/
	
		virtual void destroy() {
			for (KeyIt it(*graph); graph->valid(it); graph->next(it)) {
				erase(it);
			}
		}
	
		/** 
			The add member function should be overloaded in the subclasses.
			\e Add extends the map with the new node.
		*/
	
		virtual void add(const KeyType&) = 0;
	
		/** 
			The erase member function should be overloaded in the subclasses.
			\e Erase removes the node from the map.
		*/
	
		virtual void erase(const KeyType&) = 0;
	
		/**
			Exception class to throw at unsupported operation.
		*/
	
		class NotSupportedOperationException {};

	};
	
}

#endif
